#' Print method for SDGLM objects
#'
#' @param x An SDGLM object
#' @param ... Additional arguments
#'
#' @return The input object `x` is returned invisibly. This function is called
#'   primarily for its side effect of printing a summary of the SDGLM object
#'   to the console.
#'   
#' @export
print.SDGLM <- function(x, ...) {
  cat("Sca-MCMC fit for", x$family, "family\n")
  cat("Method:", x$method, "\n")
  cat("Beta chain dimensions:", dim(x$beta_chain), "\n")
  invisible(x)
}

#' Summary method for SDGLM objects
#'
#' @param object An SDGLM object
#' @param ... Additional arguments
#'
#' @return An object of class `summary.SDGLM`, which is a list containing:
#'   \item{coefficients}{A data frame with columns `Estimate` (posterior mean)
#'     and `Std.Error` (posterior standard deviation) for each parameter}
#'   \item{family}{Character string indicating the model family used}
#'   \item{method}{Character string indicating the mutation-rate strategy used}
#'
#' @export
summary.SDGLM <- function(object, ...) {
  # Compute posterior means
  beta_mean <- apply(object$beta_chain, 2, mean)
  beta_sd <- apply(object$beta_chain, 2, stats::sd)
  
  result <- data.frame(
    Estimate = beta_mean,
    Std.Error = beta_sd
  )
  
  structure(list(
    coefficients = result,
    family = object$family,
    method = object$method
  ), class = "summary.SDGLM")
}

#' Print method for summary.SDGLM objects
#'
#' @param x A summary.SDGLM object
#' @param ... Additional arguments
#'
#' @return The input object `x` is returned invisibly. This function is called
#'   primarily for its side effect of printing a formatted summary of the
#'   SDGLM model to the console.
#'
#' @export
print.summary.SDGLM <- function(x, ...) {
  cat("Sca-MCMC summary for", x$family, "family\n")
  cat("Method:", x$method, "\n\n")
  cat("Coefficients:\n")
  print(x$coefficients)
  invisible(x)
}
