#' Scalable Mutation-Rate Strategies for Sca-MCMC
#'
#' Computes the mutation-rate vector Q_P according to three scalable
#' schemes proposed in the reference paper.
#'
#' @param type Character, one of "ScaI", "ScaII", "ScaIII".
#' @param N Integer > 1, number of parallel chains.
#' @param L Integer > 0, length of the parameter vector.
#' @param beta.star Target parameter vector (binary or factor).
#' @param beta0 Initial parameter vector (same length as beta.star).
#'
#' @return Numeric vector of length 2, r, or N depending on type,
#'         summing to Q0 and proportional to the chosen strategy.
#'
#' @examples
#' beta.star <- c(1, 0, 1, 1, 0)
#' beta0 <- c(1, 1, 1, 0, 0)
#' mutRate("ScaII", N = 8, L = 5, beta.star, beta0)
#'
#' @export
mutRate <- function(type = c("ScaI", "ScaII", "ScaIII"),
                    N, L, beta.star, beta0) {
  type <- match.arg(type)
  dH <- sum(beta.star != beta0)  # normalized Hamming distance
  Q0 <- min(3 * dH / L, 1)      # eq. (2.3)
  
  switch(type,
         ScaI = {
           tau <- stats::runif(1)
           c(tau, 1 - tau) * Q0
         },
         ScaII = {
           r <- sample(2:(N - 1), 1)
           rep(1 / r, r) * Q0
         },
         ScaIII = {
           w <- stats::rgamma(N, shape = 1)
           w / sum(w) * Q0
         })
}
