% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelstoch_radial.R
\name{modelstoch_radial}
\alias{modelstoch_radial}
\title{Chance Constrained Radial Models}
\usage{
modelstoch_radial(datadea,
            alpha = 0.05,
            dmu_eval = NULL,
            dmu_ref = NULL,
            orientation = c("io", "oo"),
            rts = c("crs", "vrs", "nirs", "ndrs", "grs"),
            L = 1,
            U = 1,
            solver = c("alabama", "cccp", "cccp2", "slsqp"),
            give_X = TRUE,
            n_attempts_max = 5,
            maxslack = FALSE,
            weight_slack_i = 1,
            weight_slack_o = 1,
            vtrans_i = NULL,
            vtrans_o = NULL,
            compute_target = TRUE,
            returnqp = FALSE,
            silent_ud = FALSE,
            ...)
}
\arguments{
\item{datadea}{The data of class \code{deadata_stoch}, including \code{n} DMUs,
and the expected values of \code{m} inputs and \code{s} outputs.}

\item{alpha}{A value for parameter alpha.}

\item{dmu_eval}{A numeric vector containing which DMUs have to be evaluated.
If \code{NULL} (default), all DMUs are considered.}

\item{dmu_ref}{A numeric vector containing which DMUs are the evaluation
reference set.
If \code{NULL} (default), all DMUs are considered.}

\item{orientation}{A string, equal to "io" (input oriented) or "oo" (output
oriented).}

\item{rts}{A string, determining the type of returns to scale, equal to "crs" (constant),
"vrs" (variable), "nirs" (non-increasing), "ndrs" (non-decreasing) or "grs" (generalized).}

\item{L}{Lower bound for the generalized returns to scale (grs).}

\item{U}{Upper bound for the generalized returns to scale (grs).}

\item{solver}{Character string with the name of the solver used by function \code{solvecop}
from package \code{optiSolve}.}

\item{give_X}{Logical. If it is \code{TRUE}, it uses an initial vector (given by
the evaluated DMU) for the solver, except for "cccp". If it is \code{FALSE}, the initial vector is given
internally by the solver and it is usually randomly generated.}

\item{n_attempts_max}{A value with the maximum number of attempts if the solver
does not converge. Each attempt uses a different initial vector.}

\item{maxslack}{Logical. If it is \code{TRUE}, it computes the max slack solution.}

\item{weight_slack_i}{A value, vector of length \code{m}, or matrix \code{m} x \code{ne}
(where \code{ne} is the length of \code{dmu_eval}) with the weights of the input slacks
for the max slack solution.}

\item{weight_slack_o}{A value, vector of length \code{s}, or matrix \code{s} x \code{ne}
(where \code{ne} is the length of \code{dmu_eval}) with the weights of the output
slacks for the max slack solution.}

\item{vtrans_i}{Numeric vector of translation for undesirable inputs.
If \code{vtrans_i[i]} is \code{NA}, then it applies the "max + 1" translation
to the i-th undesirable input. If \code{vtrans_i} is a constant, then it applies
the same translation to all undesirable inputs. If \code{vtrans_i} is \code{NULL},
then it applies the "max + 1" translation to all undesirable inputs.}

\item{vtrans_o}{Numeric vector of translation for undesirable outputs,
analogous to \code{vtrans_i}, but applied to outputs.}

\item{compute_target}{Logical. If it is \code{TRUE}, it computes targets of the
max slack solution.}

\item{returnqp}{Logical. If it is \code{TRUE}, it returns the quadratic problems
(objective function and constraints) of stage 1.}

\item{silent_ud}{Logical, to avoid warnings related with undesirable variables.}

\item{...}{Other parameters, like the initial vector \code{X}, to be passed to the solver.}
}
\value{
A list with the results for the evaluated DMUs and other parameters
for reproducibility.
}
\description{
It solves input and output oriented chance constrained radial
DEA models under constant (CCR model), variable (BCC model),
non-increasing, non-decreasing or generalized returns to scale, based on the
model in Cooper et al. (2002). By default, models are solved in a two-stage
process (slacks are maximized).

We consider \eqn{\mathcal{D}=\left\{ \textrm{DMU}_1, \ldots ,\textrm{DMU}_n \right\} }
a set of \eqn{n} DMUs with \eqn{m} stochastic inputs and \eqn{s} stochastic outputs.
Matrices \eqn{\tilde{X}=(\tilde{x}_{ij})} and \eqn{\tilde{Y}=(\tilde{y}_{rj})}
are the input and output data matrices, respectively, where \eqn{\tilde{x}_{ij}}
and \eqn{\tilde{y}_{rj}} represent the \eqn{i}-th input and \eqn{r}-th output
of the \eqn{j}-th DMU. Moreover, we denote by \eqn{X=(x_{ij})} and \eqn{Y=(y_{rj})}
their expected values. In general, we denote vectors by bold-face letters and
they are considered as column vectors unless otherwise stated. The \eqn{0}-vector
is denoted by \eqn{\bm{0}} and the context determines its dimension.

Given \eqn{0<\alpha <1}, the first stage program for \eqn{\text{DMU}_o} with
constant returns to scale is given by
\deqn{\min \limits_{\theta, \bm{\lambda}}\quad \theta}
\deqn{\text{s.t.}\quad P\left\{ \left( \theta \tilde{\mathbf{x}}_o-\tilde{X}
\bm{\lambda}\right) _i\geq 0\right\} \geq 1-\alpha,\quad i=1,\ldots ,m,}
\deqn{P\left\{ \left( \tilde{Y}\bm{\lambda}-\tilde{\mathbf{y}}_{o}\right) _r
\geq 0\right\} \geq 1-\alpha,\quad r=1,\ldots ,s,}
\deqn{\bm{\lambda}\geq \mathbf{0},}
where \eqn{\bm{\lambda}=(\lambda_1,\ldots,\lambda_n)^\top}, \eqn{\tilde{\mathbf{x}}_o
=(\tilde{x}_{1o},\ldots,\tilde{x}_{mo})^\top} and \eqn{\tilde{\mathbf{y}}_o=
(\tilde{y}_{1o},\ldots,\tilde{y}_{so})^\top} are column vectors.
Different returns to scale can be easily considered by adding the corresponding
constraints: \eqn{\mathbf{e}\bm{\lambda}=1} (VRS), \eqn{0\leq \mathbf{e}\bm{\lambda}
\leq 1} (NIRS), \eqn{\mathbf{e}\bm{\lambda}\geq 1} (NDRS) or \eqn{L\leq \mathbf{e}
\bm{\lambda}\leq U} (GRS), with \eqn{0\leq L\leq 1} and \eqn{U\geq 1}, where
\eqn{\mathbf{e}=(1,\ldots ,1)} is a row vector.

The corresponding second stage program is given by
\deqn{\max \limits_{\bm{\lambda},\mathbf{s}^-,\mathbf{s}^+}\quad
\mathbf{w}^-\mathbf{s}^-+\mathbf{w}^+\mathbf{s}^+}
\deqn{\text{s.t.}\quad P\left\{ \left( \theta ^*\tilde{\mathbf{x}}_o-\tilde{X}
\bm{\lambda}-\mathbf{s}^-\right) _i\geq 0\right\} = 1-\alpha,\quad i=1,\ldots ,m,}
\deqn{P\left\{ \left( \tilde{Y}\bm{\lambda}-\tilde{\mathbf{y}}_{o}-\mathbf{s}^+
\right) _r\geq 0\right\} = 1-\alpha,\quad r=1,\ldots ,s,}
\deqn{\bm{\lambda}\geq \mathbf{0},\,\, \mathbf{s}^-\geq \mathbf{0},\,\,
\mathbf{s}^+\geq \mathbf{0},}
where \eqn{\theta ^*} is the optimal objective function of the first stage
program, \eqn{\mathbf{s}^-,\mathbf{s}^+} are column vectors with the slacks,
and \eqn{\mathbf{w}^-,\mathbf{w}^+} are positive row vectors with the weights
for the slacks.

The deterministic equivalents for a multivariate normal distribution of inputs/outputs
are given by
\deqn{\min \limits_{\theta, \bm{\lambda}}\quad \theta}
\deqn{\text{s.t.}\quad X\bm{\lambda}-\Phi ^{-1}(\alpha)\bm{\sigma} ^-(\theta,
\bm{\lambda}) \leq \theta \mathbf{x}_o,}
\deqn{Y\bm{\lambda}+\Phi ^{-1}(\alpha)\bm{\sigma} ^+(\bm{\lambda}) \geq \mathbf{y}_o,}
\deqn{\bm{\lambda}\geq \mathbf{0},}
and for the second stage,
\deqn{\max \limits_{\bm{\lambda},\mathbf{s}^-,\mathbf{s}^+}\quad
\mathbf{w}^-\mathbf{s}^-+\mathbf{w}^+\mathbf{s}^+}
\deqn{\text{s.t.}\quad X\bm{\lambda}+\mathbf{s}^--\Phi ^{-1}(\alpha)
\bm{\sigma} ^-(\theta ^*,\bm{\lambda}) =\theta ^* \mathbf{x}_o,}
\deqn{Y\bm{\lambda}-\mathbf{s}^++\Phi ^{-1}(\alpha)\bm{\sigma} ^+
(\bm{\lambda}) = \mathbf{y}_{o},}
\deqn{\bm{\lambda}\geq \mathbf{0},\,\, \mathbf{s}^-\geq \mathbf{0},\,\,
\mathbf{s}^+\geq \mathbf{0},}
where \eqn{\Phi } is the standard normal distribution, and
\deqn{\displaystyle \left( \sigma ^-_i\left( \theta, \bm{\lambda}\right)\right) ^2 =
\sum _{j,q=1}^n\lambda _j\lambda _q\mathrm{Cov}(\tilde{x}_{ij},\tilde{x}_{iq})
-2\theta\sum _{j=1}^n\lambda _j\mathrm{Cov}(\tilde{x}_{ij},\tilde{x}_{io})}
\deqn{+\theta ^2\mathrm{Var}(\tilde{x}_{io}),\quad i=1,\ldots ,m,}
\deqn{\displaystyle \left( \sigma ^+_r\left( \bm{\lambda}\right)\right) ^2 =
\sum _{j,q=1}^n\lambda _j\lambda _q\mathrm{Cov}(\tilde{y}_{rj},\tilde{y}_{rq})
-2\sum _{j=1}^n\lambda _j\mathrm{Cov}(\tilde{y}_{rj},\tilde{y}_{ro})}
\deqn{+\mathrm{Var}(\tilde{y}_{ro}),\quad r=1,\ldots ,s.}
}
\examples{
\donttest{
# Example 1.
library(deaR)
data("Coll_Blasco_2006")
ni <- 2 # number of inputs
no <- 2 # number of outputs
data_example <- make_deadata(datadea = Coll_Blasco_2006,
                             ni = ni,
                             no = no)
nd <- length(data_example$dmunames) # number of DMUs
var_input <- matrix(1, nrow = ni, ncol = nd)
var_output <- matrix(1, nrow = no, ncol = nd)
data_stoch <- make_deadata_stoch(datadea = data_example,
                                 var_input = var_input,
                                 var_output = var_output)
Collstoch <- modelstoch_radial(data_stoch)

# Example 2. Deterministic data with one stochastic input.
# Replication of results in El-Demerdash et al. (2013).
library(deaR)
dmunames <- c("A", "B", "C")
nd <- length(dmunames) # Number of DMUs
inputnames <- c("Professors", "Budget")
ni <- length(inputnames) # Number of Inputs
outputnames <- c("Diplomas", "Bachelors", "Masters")
no <- length(outputnames) # Number of Outputs
X <- matrix(c(5, 14, 8, 15, 7, 12),
            nrow = ni, ncol = nd, dimnames = list(inputnames, dmunames))
Y <- matrix(c(9, 4, 16, 5, 7, 10, 4, 9, 13),
            nrow = no, ncol = nd, dimnames = list(outputnames, dmunames))
datadea <- make_deadata(inputs = X,
                        outputs = Y)
covX <- array(0, dim = c(2, 3, 3))
# The 2nd input is stochastic.
# Since the corresponding 3x3 covariances matrix is symmetric, only values
# above the diagonal are necessary.
covX[2, 1, ] <- c(1.4, 0.9, 0.6)
covX[2, 2, 2:3] <- c(1.5, 0.7)
covX[2, 3, 3] <- 1.2
# Alternatively (note that values below the diagonal are ignored).
covX[2, , ] <- matrix(c(1.4, 0.9, 0.6, 0, 1.5, 0.7, 0, 0, 1.2),
                      byrow = TRUE)
datadea_stoch <- make_deadata_stoch(datadea,
                                    cov_input = covX)
res <- modelstoch_radial(datadea_stoch, rts = "vrs")
}

}
\references{
Cooper, W.W.; Deng, H.; Huang, Z.; Li, S.X. (2002). “Chance constrained programming
approaches to technical efficiencies and inefficiencies in stochastic data envelopment
analysis", Journal of the Operational Research Society, 53:12, 1347-1356.
\doi{10.1057/palgrave.jors.2601433}

El-Demerdash, B.E.; El-Khodary, I.A.; Tharwat, A.A. (2013). "Developing a Stochastic
Input Oriented Data Envelopment Analysis (SIODEA) Model", International Journal
of Advanced Computer Science and Applications, Vol.4, No. 4, 40-44.
}
\author{
\strong{Vicente Bolós} (\email{vicente.bolos@uv.es}).
\emph{Department of Business Mathematics}

\strong{Rafael Benítez} (\email{rafael.suarez@uv.es}).
\emph{Department of Business Mathematics}

\strong{Vicente Coll-Serrano} (\email{vicente.coll@uv.es}).
\emph{Quantitative Methods for Measuring Culture (MC2). Applied Economics.}

University of Valencia (Spain)
}
