/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5EApkg.h"    

herr_t
H5EA__hdr_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, const H5EA_class_t *cls,
                haddr_t obj_addr)
{
    
    H5EA_hdr_t *hdr       = NULL;    
    void       *dbg_ctx   = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(H5_addr_defined(obj_addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "unable to create fixed array debugging context");

    
    if (NULL == (hdr = H5EA__hdr_protect(f, addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to load extensible array header");

    
    Rfprintf(stream, "%*sExtensible Array Header...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Header size:", (unsigned long long)hdr->size);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Raw Element Size:", (unsigned)hdr->cparam.raw_elmt_size);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Native Element Size (on this platform):", (unsigned long long)hdr->cparam.cls->nat_elmt_size);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Log2(Max. # of elements in array):", (unsigned)hdr->cparam.max_nelmts_bits);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "# of elements in index block:", (unsigned)hdr->cparam.idx_blk_elmts);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Min. # of elements per data block:", (unsigned)hdr->cparam.data_blk_min_elmts);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Min. # of data block pointers for a super block:", (unsigned)hdr->cparam.sup_blk_min_data_ptrs);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Log2(Max. # of elements in data block page):", (unsigned)hdr->cparam.max_dblk_page_nelmts_bits);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Highest element index stored (+1):", hdr->stats.stored.max_idx_set);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of super blocks created:", hdr->stats.stored.nsuper_blks);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of data blocks created:", hdr->stats.stored.ndata_blks);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of elements 'realized':", hdr->stats.stored.nelmts);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Index Block Address:", hdr->idx_blk_addr);

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                    "unable to release extensible array debugging context");
    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__iblock_debug(H5F_t *f, haddr_t H5_ATTR_UNUSED addr, FILE *stream, int indent, int fwidth,
                   const H5EA_class_t *cls, haddr_t hdr_addr, haddr_t obj_addr)
{
    
    H5EA_hdr_t    *hdr       = NULL;    
    H5EA_iblock_t *iblock    = NULL;    
    void          *dbg_ctx   = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);
    assert(H5_addr_defined(hdr_addr));
    assert(H5_addr_defined(obj_addr));

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "unable to create extensible array debugging context");

    
    if (NULL == (hdr = H5EA__hdr_protect(f, hdr_addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to load extensible array header");

    
    assert(H5_addr_eq(hdr->idx_blk_addr, addr));

    
    if (NULL == (iblock = H5EA__iblock_protect(hdr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array index block, address = %llu",
                    (unsigned long long)hdr->idx_blk_addr);

    
    Rfprintf(stream, "%*sExtensible Array Index Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Index Block size:", (unsigned long long)iblock->size);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "# of data block addresses in index block:", (unsigned long long)iblock->ndblk_addrs);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "# of super block addresses in index block:", (unsigned long long)iblock->nsblk_addrs);

    
    if (hdr->cparam.idx_blk_elmts > 0) {
        unsigned u; 

        
        Rfprintf(stream, "%*sElements in Index Block:\n", indent, "");
        for (u = 0; u < hdr->cparam.idx_blk_elmts; u++) {
            
            if ((hdr->cparam.cls->debug)(stream, (indent + 3), MAX(0, (fwidth - 3)), (hsize_t)u,
                                         ((uint8_t *)iblock->elmts) + (hdr->cparam.cls->nat_elmt_size * u)) <
                0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "can't get element for debugging");
        } 
    }     

    
    if (iblock->ndblk_addrs > 0) {
        char     temp_str[128]; 
        unsigned u;             

        
        Rfprintf(stream, "%*sData Block Addresses in Index Block:\n", indent, "");
        for (u = 0; u < iblock->ndblk_addrs; u++) {
            
            snprintf(temp_str, sizeof(temp_str), "Address #%u:", u);
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", (indent + 3), "", MAX(0, (fwidth - 3)), temp_str,
                    iblock->dblk_addrs[u]);
        } 
    }     

    
    if (iblock->nsblk_addrs > 0) {
        char     temp_str[128]; 
        unsigned u;             

        
        Rfprintf(stream, "%*sSuper Block Addresses in Index Block:\n", indent, "");
        for (u = 0; u < iblock->nsblk_addrs; u++) {
            
            snprintf(temp_str, sizeof(temp_str), "Address #%u:", u);
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", (indent + 3), "", MAX(0, (fwidth - 3)), temp_str,
                    iblock->sblk_addrs[u]);
        } 
    }     

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                    "unable to release extensible array debugging context");
    if (iblock && H5EA__iblock_unprotect(iblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array index block");
    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__sblock_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, const H5EA_class_t *cls,
                   haddr_t hdr_addr, unsigned sblk_idx, haddr_t obj_addr)
{
    
    H5EA_hdr_t    *hdr       = NULL;    
    H5EA_sblock_t *sblock    = NULL;    
    void          *dbg_ctx   = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);
    assert(H5_addr_defined(hdr_addr));
    assert(H5_addr_defined(obj_addr));

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "unable to create extensible array debugging context");

    
    if (NULL == (hdr = H5EA__hdr_protect(f, hdr_addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to load extensible array header");

    
    
    if (NULL ==
        (sblock = H5EA__sblock_protect(hdr, (H5EA_iblock_t *)hdr, addr, sblk_idx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array super block, address = %llu",
                    (unsigned long long)addr);

    
    Rfprintf(stream, "%*sExtensible Array Super Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Super Block size:", (unsigned long long)sblock->size);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "# of data block addresses in super block:", (unsigned long long)sblock->ndblks);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "# of elements in data blocks from this super block:", (unsigned long long)sblock->dblk_nelmts);

    
    if (sblock->ndblks > 0) {
        char     temp_str[128]; 
        unsigned u;             

        
        Rfprintf(stream, "%*sData Block Addresses in Super Block:\n", indent, "");
        for (u = 0; u < sblock->ndblks; u++) {
            
            snprintf(temp_str, sizeof(temp_str), "Address #%u:", u);
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", (indent + 3), "", MAX(0, (fwidth - 3)), temp_str,
                    sblock->dblk_addrs[u]);
        } 
    }     

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                    "unable to release extensible array debugging context");
    if (sblock && H5EA__sblock_unprotect(sblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array super block");
    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblock_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, const H5EA_class_t *cls,
                   haddr_t hdr_addr, size_t dblk_nelmts, haddr_t obj_addr)
{
    
    H5EA_hdr_t    *hdr     = NULL;      
    H5EA_dblock_t *dblock  = NULL;      
    void          *dbg_ctx = NULL;      
    size_t         u;                   
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(cls);
    assert(H5_addr_defined(hdr_addr));
    assert(H5_addr_defined(obj_addr));
    assert(dblk_nelmts > 0);

    
    if (cls->crt_dbg_ctx)
        
        if (NULL == (dbg_ctx = cls->crt_dbg_ctx(f, obj_addr)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "unable to create extensible array debugging context");

    
    if (NULL == (hdr = H5EA__hdr_protect(f, hdr_addr, dbg_ctx, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL, "unable to load extensible array header");

    
    
    if (NULL == (dblock = H5EA__dblock_protect(hdr, hdr, addr, dblk_nelmts, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array data block, address = %" PRIuHADDR, addr);

    
    Rfprintf(stream, "%*sExtensible Array data Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Array class ID:", hdr->cparam.cls->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Data Block size:", (unsigned long long)dblock->size);

    
    Rfprintf(stream, "%*sElements:\n", indent, "");
    for (u = 0; u < dblk_nelmts; u++) {
        
        if ((hdr->cparam.cls->debug)(stream, (indent + 3), MAX(0, (fwidth - 3)), (hsize_t)u,
                                     ((uint8_t *)dblock->elmts) + (hdr->cparam.cls->nat_elmt_size * u)) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTGET, FAIL, "can't get element for debugging");
    } 

done:
    if (dbg_ctx && cls->dst_dbg_ctx(dbg_ctx) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTRELEASE, FAIL,
                    "unable to release extensible array debugging context");
    if (dblock && H5EA__dblock_unprotect(dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array data block");
    if (hdr && H5EA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
