% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cPCA.R
\name{cPCAplus}
\alias{cPCAplus}
\title{Contrastive PCA++ (cPCA++)
Performs Contrastive PCA++ (cPCA++) to find directions that capture variation
enriched in a "foreground" dataset relative to a "background" dataset.
This implementation follows the cPCA++ approach which directly solves the
generalized eigenvalue problem Rf v = lambda Rb v, where Rf and Rb are
the covariance matrices of the foreground and background data, centered
using the \emph{background mean}.}
\usage{
cPCAplus(
  X_f,
  X_b,
  ncomp = NULL,
  center_background = TRUE,
  lambda = 0,
  method = c("geigen", "primme", "sdiag", "corpcor"),
  strategy = c("auto", "feature", "sample"),
  verbose = getOption("multivarious.verbose", TRUE),
  sample_rank = NULL,
  sample_oversample = 10L,
  ...
)
}
\arguments{
\item{X_f}{A numeric matrix representing the foreground dataset (samples x features).}

\item{X_b}{A numeric matrix representing the background dataset (samples x features).
\code{X_f} and \code{X_b} must have the same number of features (columns).}

\item{ncomp}{Integer. The number of contrastive components to compute. Defaults to
\code{min(ncol(X_f), nrow(X_f), nrow(X_b))}, and may be further capped by the
effective background rank (especially under the sample-space strategy).}

\item{center_background}{Logical. If TRUE (default), both \code{X_f} and \code{X_b} are centered using the
column means of \code{X_b}. If FALSE, it assumes data is already appropriately centered.}

\item{lambda}{Shrinkage intensity for covariance estimation (0 <= lambda <= 1).
Defaults to 0 (no shrinkage). Uses \code{corpcor::cov.shrink}. Can help stabilize
results if \code{Rb} is ill-conditioned or singular.}

\item{method}{A character string specifying the primary computation method. Options include:
\itemize{
\item \code{"geigen"} (Default): Use \code{geneig} from the \code{geigen} package.
\item \code{"primme"}: Use \code{geneig} with the PRIMME library backend (requires special \code{geigen} build).
\item \code{"sdiag"}: Use \code{geneig} with a spectral decomposition method.
\item \code{"corpcor"}: Use a corpcor-based whitening approach followed by standard PCA.
}}

\item{strategy}{Controls the GEVD approach when \code{method} is not \code{"corpcor"}. Options include:
\itemize{
\item \code{"auto"} (Default): Chooses based on dimensions (feature vs. sample space).
\item \code{"feature"}: Forces direct computation via \verb{p x p} covariance matrices.
\item \code{"sample"}: Forces sample-space computation via SVD and a smaller GEVD (efficient for large \code{p}).
}}

\item{verbose}{Logical; if TRUE (default), prints brief status messages about strategy
selection and defaults. Set to FALSE to silence these messages.}

\item{sample_rank}{Optional integer controlling the background subspace rank used in the
sample-space strategy. If \code{NULL} (default), uses the full background rank \code{min(n_b-1, p)}.
If provided, the solver will target approximately \code{sample_rank + sample_oversample} and
will be bounded above by the full background rank.}

\item{sample_oversample}{Integer oversampling margin (default 10) applied when \code{sample_rank}
is given. Ignored when \code{sample_rank} is \code{NULL}.}

\item{...}{Additional arguments passed to the underlying computation functions
(\code{geigen::geneig} or \code{irlba::irlba} based on \code{method} and \code{strategy}).}
}
\value{
A \code{bi_projector}-like object with classes \code{c("cPCAplus", "<method_class>", "bi_projector")} containing:
\describe{
\item{v}{Loadings matrix (features x ncomp). Interpretation depends on \code{method} (see Details).}
\item{s}{Scores matrix (samples_f x ncomp).}
\item{sdev}{Vector (length ncomp). Standard deviations (sqrt of generalized eigenvalues for \code{geigen} methods, PCA std devs for \code{corpcor}).}
\item{values}{Vector (length ncomp). Generalized eigenvalues (for \code{geigen} methods) or PCA eigenvalues (for \code{corpcor}).}
\item{strategy}{The strategy used ("feature" or "sample") if method was not "corpcor".}
\item{preproc}{The initialized \code{preprocessor} object used.}
\item{method}{The computation method used.}
\item{ncomp}{The number of components computed.}
\item{nfeatures}{The number of features.}
}
}
\description{
Contrastive PCA++ (cPCA++)
Performs Contrastive PCA++ (cPCA++) to find directions that capture variation
enriched in a "foreground" dataset relative to a "background" dataset.
This implementation follows the cPCA++ approach which directly solves the
generalized eigenvalue problem Rf v = lambda Rb v, where Rf and Rb are
the covariance matrices of the foreground and background data, centered
using the \emph{background mean}.
}
\details{
\strong{Preprocessing:} Following the cPCA++ paper, if \code{center_background = TRUE}, both \code{X_f} and \code{X_b}
are centered by subtracting the column means calculated \emph{only} from the background data \code{X_b}.
This is crucial for isolating variance specific to \code{X_f}.

\strong{Core Algorithm (methods "geigen", "primme", "sdiag", strategy="feature"):}
\enumerate{
\item Center \code{X_f} and \code{X_b} using the mean of \code{X_b}.
\item Compute potentially shrunk \eqn{p \times p} covariance matrices \code{Rf} (from centered \code{X_f}) and \code{Rb} (from centered \code{X_b}) using \code{corpcor::cov.shrink}.
\item Solve the generalized eigenvalue problem \verb{Rf v = lambda Rb v} for the top \code{ncomp} eigenvectors \code{v} using \code{geigen::geneig}. These eigenvectors are the contrastive principal components (loadings).
\item Compute scores by projecting the centered foreground data onto the eigenvectors: \code{S = X_f_centered \%*\% v}.
}

\strong{Core Algorithm (Large-D / Sample Space Strategy, strategy="sample"):}
When \eqn{p \gg n}, forming \eqn{p \times p} matrices \code{Rf} and \code{Rb} is infeasible. The "sample" strategy follows cPCA++ §3.2:
\enumerate{
\item Center \code{X_f} and \code{X_b} using the mean of \code{X_b}.
\item Compute the SVD of centered \eqn{X_b = Ub Sb Vb^T} (using \code{irlba} for efficiency).
\item Project centered \code{X_f} into the background's principal subspace: \code{Zf = X_f_centered \%*\% Vb}.
\item Form small \eqn{r \times r} matrices: \code{Rf_small = cov(Zf)} and \code{Rb_small = (1/(n_b-1)) * Sb^2}.
\item Solve the small \eqn{r \times r} GEVD: \verb{Rf_small w = lambda Rb_small w} using \code{geigen::geneig}.
\item Lift eigenvectors back to feature space: \code{v = Vb \%*\% w}.
\item Compute scores: \code{S = X_f_centered \%*\% v}.
}

\strong{Alternative Algorithm (method "corpcor"):}
\enumerate{
\item Center \code{X_f} and \code{X_b} using the mean of \code{X_b}.
\item Compute \code{Rb} and its inverse square root \code{Rb_inv_sqrt}.
\item Whiten the foreground data: \code{X_f_whitened = X_f_centered \%*\% Rb_inv_sqrt}.
\item Perform standard PCA (\code{stats::prcomp}) on \code{X_f_whitened}.
\item The returned \code{v} and \code{s} are the loadings and scores \emph{in the whitened space}. The loadings are \emph{not} the generalized eigenvectors \code{v}. A specific class \code{corpcor_pca} is added to signal this.
}
}
\examples{
# Simulate data where foreground has extra variance in first few dimensions
set.seed(123)
n_f <- 100
n_b <- 150
n_features <- 50

# Background: standard normal noise
X_b <- matrix(rnorm(n_b * n_features), nrow=n_b, ncol=n_features)
colnames(X_b) <- paste0("Feat_", 1:n_features)

# Foreground: background noise + extra variance in first 5 features
X_f_signal <- matrix(rnorm(n_f * 5, mean=0, sd=2), nrow=n_f, ncol=5)
X_f_noise <- matrix(rnorm(n_f * (n_features-5)), nrow=n_f, ncol=n_features-5)
X_f <- cbind(X_f_signal, X_f_noise) + matrix(rnorm(n_f * n_features), nrow=n_f, ncol=n_features)
colnames(X_f) <- paste0("Feat_", 1:n_features)
rownames(X_f) <- paste0("SampleF_", 1:n_f)

# Apply cPCA++ (requires geigen and corpcor packages)
# install.packages(c("geigen", "corpcor"))
if (requireNamespace("geigen", quietly = TRUE) && requireNamespace("corpcor", quietly = TRUE)) {
  # Assuming helper constructors like bi_projector are available
  # library(multivarious) 

  res_cpca_plus <- cPCAplus(X_f, X_b, ncomp = 5, method = "geigen")

  # Scores for the foreground data (samples x components)
  print(head(res_cpca_plus$s))

  # Loadings (contrastive directions) (features x components)
  print(head(res_cpca_plus$v))
}

\donttest{
# Plot example (slow graphics)
if (requireNamespace("geigen", quietly = TRUE) && requireNamespace("corpcor", quietly = TRUE)) {
  set.seed(123)
  X_b <- matrix(rnorm(150 * 50), nrow=150, ncol=50)
  X_f <- cbind(matrix(rnorm(100*5, sd=2), 100, 5), matrix(rnorm(100*45), 100, 45))
  res <- cPCAplus(X_f, X_b, ncomp = 5, method = "geigen")
  plot(res$s[, 1], res$s[, 2],
       xlab = "Contrastive Component 1", ylab = "Contrastive Component 2",
       main = "cPCA++ Scores")
}
}

}
\references{
Salloum, R., Kuo, C. C. J. (2022). cPCA++: An efficient method for contrastive feature learning. Pattern Recognition, 124, 108378. (Algorithm 1)
}
