# Register global variables for CRAN compliance
utils::globalVariables(NULL)

#' Two-Phase Experimental Design Development
#'
#' @description
#' `TPDDev()` constructs a two-phase experimental design layout,
#' computes the information matrices for residual, direct, and interaction effects,
#' and calculates the efficiency factor.
#'
#' @param d1 Numeric matrix. Phase-I layout (rows = blocks, columns = plots per block).
#' @param base_d2 Numeric matrix. Base Phase-II layout (rows = blocks, columns = plots per block).
#'
#' @return A list containing design parameters, combined two-phase design layout,
#'         information matrices, and efficiency factor.
#'
#' @details
#' Steps performed:
#' \enumerate{
#'   \item Checks that `ncol(d1)` equals `nrow(base_d2)`.
#'   \item Expands the Phase-II layout for each Phase-I block.
#'   \item Builds the combined layout and treatment-incidence matrices.
#'   \item Computes block-incidence matrix and overall mean matrix.
#'   \item Derives overall C-matrix for factorial treatment effects.
#'   \item Projects C-matrix to obtain residual, direct, and interaction effect matrices.
#'   \item Calculates the efficiency factor of the design.
#' }
#'
#' @examples
#' d1 <- matrix(c(
#'   1,2,3,4,5,6,7,
#'   2,3,4,5,6,7,8,
#'   3,4,5,6,7,8,1,
#'   4,5,6,7,8,1,2,
#'   5,6,7,8,1,2,3,
#'   6,7,8,1,2,3,4,
#'   7,8,1,2,3,4,5,
#'   8,1,2,3,4,5,6
#' ), nrow = 8, byrow = TRUE)
#'
#' base_d2 <- matrix(c(
#'   1,2,4,
#'   2,3,5,
#'   3,4,6,
#'   4,5,7,
#'   5,6,1,
#'   6,7,2,
#'   7,1,3
#' ), ncol = 3, byrow = TRUE)
#'
#' result <- TPDDev(d1, base_d2)
#' result$efficiency_factor
#' result$C_dir
#'
#' @importFrom MASS ginv
#' @import Matrix
#' @export
TPDDev <- function(d1, base_d2) {
  # --- Check inputs ---
  if (ncol(d1) != nrow(base_d2))
    stop("Input requirement violated: ncol(d1) must equal nrow(base_d2).")

  # --- Dimensions ---
  b1 <- nrow(d1); k1 <- ncol(d1)
  b2 <- nrow(base_d2); k2 <- ncol(base_d2)

  # --- Expand Phase-II layout ---
  d2 <- do.call(rbind, replicate(b1, base_d2, simplify = FALSE))

  # --- Combined layout ---
  d11 <- matrix(as.vector(t(d1)), ncol = 1)
  d_combined <- cbind(d11, d2)

  # --- Factorial treatment incidence ---
  v1 <- max(d1); v2 <- max(d2)
  tot_trt <- v1 * v2
  nObs <- nrow(d2) * ncol(d2)

  uniq1 <- sort(unique(as.vector(d1)))
  uniq2 <- sort(unique(as.vector(d2)))
  facto <- matrix(NA, nrow = tot_trt, ncol = 2)
  k <- 1
  for (i in 1:v1) for (j in 1:v2) {
    facto[k, ] <- c(uniq1[i], uniq2[j]); k <- k + 1
  }

  # --- Treatment incidence matrix ---
  trt_d12 <- matrix(0, nrow = nObs, ncol = tot_trt)
  obs <- 1
  for (i in 1:nrow(d_combined)) {
    p1 <- d_combined[i, 1]
    for (j in 2:ncol(d_combined)) {
      p2 <- d_combined[i, j]
      match <- which(facto[,1]==p1 & facto[,2]==p2)
      trt_d12[obs, match] <- 1
      obs <- obs + 1
    }
  }

  # --- Block incidence matrix ---
  block_new <- matrix(0, nrow = nObs, ncol = nrow(d2))
  obs <- 1
  for (i in 1:nrow(d2)) for (j in 1:ncol(d2)) {
    block_new[obs, i] <- 1; obs <- obs + 1
  }

  # --- Overall mean ---
  m <- matrix(1, nrow = nObs, ncol = 1)

  # --- C-matrix ---
  x1 <- trt_d12; x2 <- cbind(block_new, m)
  C_mat <- round(
    t(x1) %*% x1 - t(x1) %*% x2 %*%
      MASS::ginv(t(x2) %*% x2) %*% t(x2) %*% x1,
    7
  )

  # --- Projection matrices ---
  p1 <- matrix(1, nrow=v1-1, ncol=v1)
  for(i in 1:(v1-1)) { p1[i,i+1]<- -i; if(i+2<=v1) p1[i,(i+2):v1]<- p1[i,(i+2):v1]-1 }
  p2 <- matrix(1, nrow=v2-1, ncol=v2)
  for(i in 1:(v2-1)) { p2[i,i+1]<- -i; if(i+2<=v2) p2[i,(i+2):v2]<- p2[i,(i+2):v2]-1 }
  p10 <- matrix(1,nrow=v1,ncol=1); p20 <- matrix(1,nrow=v2,ncol=1)
  p11 <- kronecker(p1, t(p20)); p22 <- kronecker(t(p10), p2); p12 <- kronecker(p1,p2)
  Np11 <- p11/sqrt(rowSums(p11^2)); Np22 <- p22/sqrt(rowSums(p22^2)); Np12 <- p12/sqrt(rowSums(p12^2))

  # --- C-matrices for effects ---
  C_res <- round(Np11 %*% C_mat %*% t(Np11),5)
  C_dir <- round(Np22 %*% C_mat %*% t(Np22),5)
  C_int <- round(Np12 %*% C_mat %*% t(Np12),5)

  # --- Efficiency Factor ---
  efficiency_factor <- round(100*(v2*(k2-1))/(k2*(v2-1)),4)

  # --- Return ---
  list(
    v1=v1, v2=v2, k2=k2,
    d_combined=d_combined,
    block_incidence=block_new,
    facto=facto,
    C_mat=C_mat,
    C_res=C_res,
    C_dir=C_dir,
    C_int=C_int,
    efficiency_factor=efficiency_factor
  )
}
