% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FRR_Dist.R
\name{fast_distance}
\alias{fast_distance}
\title{JAX-accelerated distance calculations}
\usage{
fast_distance(
  A,
  B = NULL,
  metric = c("euclidean", "sqeuclidean", "manhattan", "chebyshev", "cosine", "minkowski",
    "mahalanobis"),
  p = 2,
  weights = NULL,
  cov_inv = NULL,
  approximate_inv = TRUE,
  squared = FALSE,
  row_batch_size = NULL,
  as_dist = FALSE,
  return_type = "R",
  verbose = FALSE,
  conda_env = "fastrerandomize_env",
  conda_env_required = TRUE
)
}
\arguments{
\item{A}{A numeric matrix with rows as observations and columns as features.}

\item{B}{Optional numeric matrix with the same number of columns as \code{A}.
If \code{NULL}, distances are computed within \code{A} (i.e., \eqn{n \times n}).}

\item{metric}{Character; one of
\code{"euclidean"}, \code{"sqeuclidean"}, \code{"manhattan"},
\code{"chebyshev"}, \code{"cosine"}, \code{"minkowski"}, \code{"mahalanobis"}.
Default is \code{"euclidean"}.}

\item{p}{Numeric order for Minkowski distance (must be \eqn{>0}). Default is \code{2}.}

\item{weights}{Optional numeric vector of length \code{ncol(A)} with nonnegative
feature weights. Used for \code{"minkowski"} and \code{"manhattan"} (the latter is
equivalent to Minkowski with \code{p = 1}).}

\item{cov_inv}{Optional inverse covariance matrix (p x p) for Mahalanobis (ignored
if \code{approximate_inv = TRUE}). If not supplied and \code{approximate_inv = FALSE},
it is estimated from \code{rbind(A, B)} and inverted in JAX.}

\item{approximate_inv}{Logical; if \code{TRUE} and \code{metric = "mahalanobis"},
uses a diagonal inverse (reciprocal variances) for speed and robustness.
Default \code{TRUE}.}

\item{squared}{Logical; if \code{TRUE}, return squared distances when supported
(\code{"euclidean"} and \code{"mahalanobis"}). Ignored for other metrics.
Default \code{FALSE}.}

\item{row_batch_size}{Optional integer; number of rows of \code{A} to process per batch.
If \code{NULL}, a safe size is chosen automatically.}

\item{as_dist}{Logical; if \code{TRUE} and \code{B} is \code{NULL}, return a base
\code{\link[stats]{dist}} object (for symmetric metrics). Default \code{FALSE}.}

\item{return_type}{Either \code{"R"} (convert to base R matrix/\code{dist}) or
\code{"jax"} (return a JAX array). Default \code{"R"}.}

\item{verbose}{Logical; print batching progress. Default \code{FALSE}.}

\item{conda_env}{Character; conda environment name used by \code{reticulate}.
Default \code{"fastrerandomize_env"}.}

\item{conda_env_required}{Logical; whether the specified conda environment must be
used. Default \code{TRUE}.}
}
\value{
An \eqn{n \times m} distance matrix in the format specified by \code{return_type}.
  If \code{as_dist = TRUE} and \code{B = NULL} (symmetric case), returns a
  \code{\link[stats]{dist}} object.
}
\description{
Compute pairwise distances between the rows of one matrix \code{A} or two matrices
\code{A} and \code{B}, using JAX-backed, JIT-compiled kernels. Supports common metrics:
Euclidean, squared Euclidean, Manhattan, Chebyshev, Cosine, Minkowski (with optional
feature weights), and Mahalanobis (with full or diagonal inverse covariance).

The function automatically batches computations to avoid excessive device memory use.
}
\details{
- **Mahalanobis**: with \code{approximate_inv = TRUE}, the diagonal of the pooled
  covariance is used (variance stabilizer); otherwise a full inverse covariance is used.
- **Weighted distances**: supply \code{weights} (length \code{p}) for
  \code{"minkowski"} and \code{"manhattan"} (the latter uses \code{p = 1}).
- Computations run in float32 and are JIT-compiled with JAX; where applicable,
  GPU/Metal/CPU device selection follows your existing backend.
}
\examples{
\dontrun{
# Simple Euclidean within-matrix distances (returns an n x n matrix)
X <- matrix(rnorm(50 * 8), 50, 8)
D <- fast_distance(X, metric = "euclidean")

# Cosine distance between two sets
A <- matrix(rnorm(100 * 16), 100, 16)
B <- matrix(rnorm(120 * 16), 120, 16)
Dcos <- fast_distance(A, B, metric = "cosine")

# Minkowski with p = 3 and feature weights
w <- runif(ncol(A))
Dm3 <- fast_distance(A, B, metric = "minkowski", p = 3, weights = w)

# Mahalanobis (diagonal approx, fast & robust)
Dmah_diag <- fast_distance(X, metric = "mahalanobis", approximate_inv = TRUE)

# Mahalanobis with full inverse (computed internally)
Dmah_full <- fast_distance(X, metric = "mahalanobis", approximate_inv = FALSE)

# Return a base R 'dist' object
D_dist <- fast_distance(X, metric = "euclidean", as_dist = TRUE)
}

}
\seealso{
\code{\link[stats]{dist}}
}
