/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5DSprivate.h"
#include "H5LTprivate.h"
#include "H5IMprivate.h"
#include "H5TBprivate.h"

static herr_t H5DS_is_reserved(hid_t did, bool *is_reserved);

herr_t
H5DSwith_new_ref(hid_t obj_id, bool *with_new_ref)
{
    bool config_flag = false;
    bool native      = false;

    if (!with_new_ref)
        return FAIL;

    if (H5VLobject_is_native(obj_id, &native) < 0)
        return FAIL;

#ifdef H5_DIMENSION_SCALES_WITH_NEW_REF
    config_flag = true;
#endif

    *with_new_ref = (config_flag || !native);

    return SUCCEED;
}

herr_t
H5DSset_scale(hid_t dsid, const char *dimname)
{
    htri_t     has_dimlist;
    H5I_type_t it;

    
    
    if ((it = H5Iget_type(dsid)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    

    
    if ((has_dimlist = H5Aexists(dsid, DIMENSION_LIST)) < 0)
        return FAIL;
    if (has_dimlist > 0)
        return FAIL;

    

    if (H5LT_set_attribute_string(dsid, "CLASS", DIMENSION_SCALE_CLASS) < 0)
        return FAIL;

    if (dimname != NULL) {
        if (H5LT_set_attribute_string(dsid, "NAME", dimname) < 0)
            return FAIL;
    }

    return SUCCEED;
}

herr_t
H5DSattach_scale(hid_t did, hid_t dsid, unsigned int idx)
{
    htri_t   has_dimlist;
    htri_t   has_reflist;
    int      is_ds;
    hssize_t nelmts;
    hid_t    sid, sid_w;             
    hid_t    tid  = H5I_INVALID_HID; 
    hid_t    ntid = H5I_INVALID_HID; 
    hid_t    aid  = H5I_INVALID_HID; 
    int      rank;                   
    hsize_t  dims[1];                

    ds_list_t  dsl;          
    ds_list_t *dsbuf = NULL; 
    ds_list_t *dsbuf_w =
        NULL; 
    hobj_ref_t ref_to_ds = HADDR_UNDEF; 
    hobj_ref_t ref_j;                   

    
    nds_list_t  ndsl;
    nds_list_t *ndsbuf     = NULL;
    nds_list_t *ndsbuf_w   = NULL;
    H5R_ref_t   nref_to_ds = {0};
    H5R_ref_t   nref_j;
    bool        is_new_ref;

    hvl_t      *buf = NULL; 
    hid_t       dsid_j;     
    H5O_info2_t oi1, oi2;
    H5I_type_t  it1, it2;
    int         i;
    size_t      len;
    int         found_ds = 0;
    htri_t      is_scale;
    bool        is_reserved;

    

    if ((is_scale = H5DSis_scale(did)) < 0)
        return FAIL;

    
    if (is_scale == 1)
        return FAIL;

    
    if (H5Oget_info3(did, &oi1, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (H5Oget_info3(dsid, &oi2, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (oi1.fileno == oi2.fileno) {
        int token_cmp;

        if (H5Otoken_cmp(did, &oi1.token, &oi2.token, &token_cmp) < 0)
            return FAIL;
        if (!token_cmp)
            return FAIL;
    } 

    

    if (H5DSwith_new_ref(did, &is_new_ref) < 0)
        return FAIL;

    
    if ((it1 = H5Iget_type(did)) < 0)
        return FAIL;
    if ((it2 = H5Iget_type(dsid)) < 0)
        return FAIL;

    if (H5I_DATASET != it1 || H5I_DATASET != it2)
        return FAIL;

    
    if (H5Aexists(dsid, DIMENSION_LIST) > 0)
        return FAIL;

    
    if (H5DS_is_reserved(did, &is_reserved) < 0)
        return FAIL;
    if (is_reserved == true)
        return FAIL;

    

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (rank == 0)
        rank = 1;

    
    if (H5Sclose(sid) < 0)
        return FAIL;

    
    if (idx > (unsigned)rank - 1)
        return FAIL;

    
    if (is_new_ref) {
        
        if (H5Rcreate_object(dsid, ".", H5P_DEFAULT, &nref_to_ds) < 0)
            return FAIL;
        
        if (H5Rcreate_object(did, ".", H5P_DEFAULT, &ndsl.ref) < 0)
            return FAIL;
    }
    else {
        
        if (H5Rcreate(&ref_to_ds, dsid, ".", H5R_OBJECT, (hid_t)-1) < 0)
            return FAIL;

        
        if (H5Rcreate(&dsl.ref, did, ".", H5R_OBJECT, (hid_t)-1) < 0)
            return FAIL;
    }
    
    if ((has_dimlist = H5Aexists(did, DIMENSION_LIST)) < 0)
        return FAIL;

    
    if (has_dimlist == 0) {

        dims[0] = (hsize_t)rank;

        
        if ((sid = H5Screate_simple(1, dims, NULL)) < 0)
            return FAIL;

        
        if (is_new_ref) {
            if ((tid = H5Tvlen_create(H5T_STD_REF)) < 0)
                goto out;
        }
        else {
            if ((tid = H5Tvlen_create(H5T_STD_REF_OBJ)) < 0)
                goto out;
        }
        
        if ((aid = H5Acreate2(did, DIMENSION_LIST, tid, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));
        if (buf == NULL)
            goto out;

        for (i = 0; i < rank; i++) {
            buf[i].len = 0;
            buf[i].p   = NULL;
        }

        
        buf[idx].len = 1;
        if (is_new_ref) {
            buf[idx].p                   = malloc(1 * sizeof(H5R_ref_t));
            ((H5R_ref_t *)buf[idx].p)[0] = nref_to_ds;
        }
        else {
            buf[idx].p                    = malloc(1 * sizeof(hobj_ref_t));
            ((hobj_ref_t *)buf[idx].p)[0] = ref_to_ds;
        }
        
        if (H5Awrite(aid, tid, buf) < 0)
            goto out;

        
        if (is_new_ref) {
            if (H5Rdestroy(&nref_to_ds) < 0)
                goto out;
        }
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(buf[idx].p);
        buf[idx].p = NULL;
        free(buf);
        buf = NULL;
    }

    
    else if (has_dimlist > 0) {
        if ((aid = H5Aopen(did, DIMENSION_LIST, H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        
        buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));
        if (buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, buf) < 0)
            goto out;

        
        
        for (i = 0; i < (int)buf[idx].len; i++) {
            
            if (is_new_ref) {
                nref_j = ((H5R_ref_t *)buf[idx].p)[i];

                
                if ((dsid_j = H5Ropen_object(&nref_j, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                    goto out;
            }
            else {
                ref_j = ((hobj_ref_t *)buf[idx].p)[i];

                
                if ((dsid_j = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref_j)) < 0)
                    goto out;
            }
            
            if (H5Oget_info3(dsid, &oi1, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (H5Oget_info3(dsid_j, &oi2, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (oi1.fileno == oi2.fileno) {
                int token_cmp;

                if (H5Otoken_cmp(did, &oi1.token, &oi2.token, &token_cmp) < 0)
                    goto out;
                if (!token_cmp)
                    found_ds = 1;
            } 

            
            if (H5Dclose(dsid_j) < 0)
                goto out;
        } 

        if (found_ds == 0) {
            
            if (buf[idx].len > 0) {
                buf[idx].len++;
                len = buf[idx].len;
                if (is_new_ref) {
                    buf[idx].p                         = realloc(buf[idx].p, len * sizeof(H5R_ref_t));
                    ((H5R_ref_t *)buf[idx].p)[len - 1] = nref_to_ds;
                }
                else {
                    buf[idx].p                          = realloc(buf[idx].p, len * sizeof(hobj_ref_t));
                    ((hobj_ref_t *)buf[idx].p)[len - 1] = ref_to_ds;
                }
            } 
            else {
                
                buf[idx].len = 1;
                if (is_new_ref) {
                    buf[idx].p                   = malloc(sizeof(H5R_ref_t));
                    ((H5R_ref_t *)buf[idx].p)[0] = nref_to_ds;
                }
                else {
                    buf[idx].p                    = malloc(sizeof(hobj_ref_t));
                    ((hobj_ref_t *)buf[idx].p)[0] = ref_to_ds;
                }
            } 
        }     
        else {
            if (is_new_ref && H5Rdestroy(&nref_to_ds) < 0)
                goto out;
        }

        
        if (H5Awrite(aid, tid, buf) < 0)
            goto out;

        
        if (H5Treclaim(tid, sid, H5P_DEFAULT, buf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(buf);
        buf = NULL;
    } 

    

    
    if ((has_reflist = H5Aexists(dsid, REFERENCE_LIST)) < 0)
        goto out;

    
    if (has_reflist == 0) {
        dims[0] = 1;

        
        if ((sid = H5Screate_simple(1, dims, NULL)) < 0)
            goto out;

        
        if (is_new_ref) {
            if ((tid = H5Tcreate(H5T_COMPOUND, sizeof(nds_list_t))) < 0)
                goto out;
            if (H5Tinsert(tid, "dataset", HOFFSET(nds_list_t, ref), H5T_STD_REF) < 0)
                goto out;
            if (H5Tinsert(tid, "dimension", HOFFSET(nds_list_t, dim_idx), H5T_NATIVE_UINT) < 0)
                goto out;
        }
        else {
            if ((tid = H5Tcreate(H5T_COMPOUND, sizeof(ds_list_t))) < 0)
                goto out;
            if (H5Tinsert(tid, "dataset", HOFFSET(ds_list_t, ref), H5T_STD_REF_OBJ) < 0)
                goto out;
            if (H5Tinsert(tid, "dimension", HOFFSET(ds_list_t, dim_idx), H5T_NATIVE_UINT) < 0)
                goto out;
        }

        
        if ((aid = H5Acreate2(dsid, REFERENCE_LIST, tid, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        if (is_new_ref) {
            ndsl.dim_idx = idx;
            if (H5Awrite(aid, tid, &ndsl) < 0)
                goto out;
            if (H5Rdestroy(&ndsl.ref) < 0)
                goto out;
        }
        else {
            dsl.dim_idx = idx;
            if (H5Awrite(aid, tid, &dsl) < 0)
                goto out;
        }
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
    } 

    
    else if (has_reflist > 0) {
        hid_t tmp_id; 
        int   j;

        if ((aid = H5Aopen(dsid, REFERENCE_LIST, H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        
        if ((ntid = H5Tget_native_type(tid, H5T_DIR_ASCEND)) < 0)
            goto out;

        
        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        if ((nelmts = H5Sget_simple_extent_npoints(sid)) < 0)
            goto out;

        nelmts++;
        if (is_new_ref) {
            ndsbuf = (nds_list_t *)malloc((size_t)nelmts * sizeof(nds_list_t));
            if (ndsbuf == NULL)
                goto out;
            if (H5Aread(aid, ntid, ndsbuf) < 0)
                goto out;
        }
        else {
            dsbuf = (ds_list_t *)malloc((size_t)nelmts * sizeof(ds_list_t));
            if (dsbuf == NULL)
                goto out;
            if (H5Aread(aid, ntid, dsbuf) < 0)
                goto out;
        }

        
        if (H5Aclose(aid) < 0)
            goto out;

        

        

        if (is_new_ref) {
            ndsbuf_w = (nds_list_t *)malloc((size_t)nelmts * sizeof(nds_list_t));
            if (ndsbuf_w == NULL)
                goto out;
        }
        else {
            dsbuf_w = (ds_list_t *)malloc((size_t)nelmts * sizeof(ds_list_t));
            if (dsbuf_w == NULL)
                goto out;
        }
        
        for (j = 0; j < nelmts - 1; j++) {
            if (is_new_ref) {
                ndsbuf_w[j].dim_idx = ndsbuf[j].dim_idx;
                tmp_id              = H5Ropen_object(&ndsbuf[j].ref, H5P_DEFAULT, H5P_DEFAULT);
                if (tmp_id < 0)
                    goto out;
                if (H5Rcreate_object(tmp_id, ".", H5P_DEFAULT, &ndsbuf_w[j].ref) < 0) {
                    H5Dclose(tmp_id);
                    goto out;
                }
            }
            else {
                dsbuf_w[j] = dsbuf[j];
            }
        }
        
        if (is_new_ref) {
            ndsl.dim_idx         = idx;
            ndsbuf_w[nelmts - 1] = ndsl;
        }
        else {
            dsl.dim_idx         = idx;
            dsbuf_w[nelmts - 1] = dsl;
        }

        
        if (H5Adelete(dsid, REFERENCE_LIST) < 0)
            goto out;

        
        dims[0] = (hsize_t)nelmts;

        if ((sid_w = H5Screate_simple(1, dims, NULL)) < 0)
            goto out;

        
        if ((aid = H5Acreate2(dsid, REFERENCE_LIST, tid, sid_w, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        if (is_new_ref) {
            if (H5Awrite(aid, ntid, ndsbuf_w) < 0)
                goto out;
            if (H5Treclaim(tid, sid, H5P_DEFAULT, ndsbuf_w) < 0)
                goto out;
        }
        else {
            if (H5Awrite(aid, ntid, dsbuf_w) < 0)
                goto out;
            if (H5Treclaim(tid, sid, H5P_DEFAULT, dsbuf_w) < 0)
                goto out;
        }
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Sclose(sid_w) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        if (H5Tclose(ntid) < 0)
            goto out;
        if (is_new_ref) {
            free(ndsbuf);
            dsbuf = NULL;
            free(ndsbuf_w);
            dsbuf = NULL;
        }
        else {
            free(dsbuf);
            dsbuf = NULL;
            free(dsbuf_w);
            dsbuf = NULL;
        }
    } 

    

    if ((is_ds = H5DSis_scale(dsid)) < 0)
        return FAIL;

    if (is_ds == 0) {
        if (H5LT_set_attribute_string(dsid, "CLASS", DIMENSION_SCALE_CLASS) < 0)
            return FAIL;
    }

    return SUCCEED;

    
out:
    if (buf)
        free(buf);
    if (dsbuf)
        free(dsbuf);
    if (dsbuf_w)
        free(dsbuf_w);

    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(ntid);
        H5Tclose(tid);
    }
    H5E_END_TRY
    return FAIL;
}

herr_t
H5DSdetach_scale(hid_t did, hid_t dsid, unsigned int idx)
{
    htri_t       has_dimlist;
    htri_t       has_reflist;
    hssize_t     nelmts;
    hid_t        dsid_j;                  
    hid_t        did_i;                   
    hid_t        sid   = H5I_INVALID_HID; 
    hid_t        sid_w = H5I_INVALID_HID; 
    hid_t        tid   = H5I_INVALID_HID; 
    hid_t        ntid  = H5I_INVALID_HID; 
    hid_t        aid   = H5I_INVALID_HID; 
    int          rank;                    
    nds_list_t  *ndsbuf   = NULL;         
    nds_list_t  *ndsbuf_w = NULL; 
    ds_list_t   *dsbuf    = NULL; 
    ds_list_t   *dsbuf_w  = NULL; 
    hsize_t      dims[1];         
    H5R_ref_t    nref;
    hobj_ref_t   ref;        
    hvl_t       *buf = NULL; 
    int          i;
    size_t       j;
    hssize_t     ii;
    H5O_info2_t  did_oi, dsid_oi, tmp_oi;
    int          found_dset = 0, found_ds = 0;
    int          have_ds = 0;
    htri_t       is_scale;
    bool         is_new_ref;
    unsigned int tmp_idx;
    hid_t        tmp_id;

    

    

    if (H5I_DATASET != H5Iget_type(did) || H5I_DATASET != H5Iget_type(dsid))
        return FAIL;

    if ((is_scale = H5DSis_scale(did)) < 0)
        return FAIL;

    
    if (is_scale == 1)
        return FAIL;

    
    if (H5Oget_info3(did, &did_oi, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (H5Oget_info3(dsid, &dsid_oi, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (did_oi.fileno == dsid_oi.fileno) {
        int token_cmp;

        if (H5Otoken_cmp(did, &did_oi.token, &dsid_oi.token, &token_cmp) < 0)
            return FAIL;
        if (!token_cmp)
            return FAIL;
    } 

    
    if (H5DSwith_new_ref(did, &is_new_ref) < 0)
        return FAIL;

    

    
    if ((has_dimlist = H5Aexists(did, DIMENSION_LIST)) < 0)
        return FAIL;
    if (has_dimlist == 0)
        return FAIL;

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        return FAIL;

    
    if (idx > (unsigned)rank - 1)
        return FAIL;

    

    
    if ((has_reflist = H5Aexists(dsid, REFERENCE_LIST)) < 0)
        return FAIL;
    if (has_reflist == 0)
        return FAIL;

    

    if ((aid = H5Aopen(did, DIMENSION_LIST, H5P_DEFAULT)) < 0)
        return FAIL;

    if ((tid = H5Aget_type(aid)) < 0)
        goto out;

    if ((sid = H5Aget_space(aid)) < 0)
        goto out;

    
    buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));
    if (buf == NULL)
        goto out;

    
    if (H5Aread(aid, tid, buf) < 0)
        goto out;

    
    if (buf[idx].len > 0) {
        for (j = 0; j < buf[idx].len; j++) {
            if (is_new_ref) {
                
                nref = ((H5R_ref_t *)buf[idx].p)[j];

                
                if ((dsid_j = H5Ropen_object(&nref, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                    goto out;
            }
            else {
                
                ref = ((hobj_ref_t *)buf[idx].p)[j];

                
                if ((dsid_j = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref)) < 0)
                    goto out;
            }
            
            if (H5Oget_info3(dsid_j, &tmp_oi, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (H5Dclose(dsid_j) < 0)
                goto out;

            
            if (dsid_oi.fileno == tmp_oi.fileno) {
                int token_cmp;

                if (H5Otoken_cmp(did, &dsid_oi.token, &tmp_oi.token, &token_cmp) < 0)
                    goto out;
                if (!token_cmp) {
                    

                    size_t len = buf[idx].len;

                    if (j < len - 1) {
                        if (is_new_ref) {
                            ((H5R_ref_t *)buf[idx].p)[j] = ((H5R_ref_t *)buf[idx].p)[len - 1];
                        }
                        else {
                            ((hobj_ref_t *)buf[idx].p)[j] = ((hobj_ref_t *)buf[idx].p)[len - 1];
                        }
                    }
                    len = --buf[idx].len;
                    if (len == 0) {
                        free(buf[idx].p);
                        buf[idx].p = NULL;
                    }
                    
                    found_ds = 1;
                    break;
                } 
            }     
        }         
    }             

    
    if (found_ds == 0)
        goto out;

    
    for (i = 0; i < rank; i++) {
        if (buf[i].len > 0) {
            have_ds = 1;
            break;
        }
    }
    if (have_ds) {
        if (H5Awrite(aid, tid, buf) < 0)
            goto out;
    }
    else {
        if (H5Adelete(did, DIMENSION_LIST) < 0)
            goto out;
    }

    
    if (H5Treclaim(tid, sid, H5P_DEFAULT, buf) < 0)
        goto out;
    if (H5Sclose(sid) < 0)
        goto out;
    if (H5Tclose(tid) < 0)
        goto out;
    if (H5Aclose(aid) < 0)
        goto out;

    free(buf);
    buf = NULL;

    

    if ((aid = H5Aopen(dsid, REFERENCE_LIST, H5P_DEFAULT)) < 0)
        goto out;

    if ((tid = H5Aget_type(aid)) < 0)
        goto out;

    
    if ((ntid = H5Tget_native_type(tid, H5T_DIR_ASCEND)) < 0)
        goto out;

    
    if ((sid = H5Aget_space(aid)) < 0)
        goto out;

    if ((nelmts = H5Sget_simple_extent_npoints(sid)) < 0)
        goto out;

    if (is_new_ref) {
        ndsbuf = (nds_list_t *)malloc((size_t)nelmts * sizeof(nds_list_t));
        if (ndsbuf == NULL)
            goto out;
        if (H5Aread(aid, ntid, ndsbuf) < 0)
            goto out;
        ndsbuf_w = (nds_list_t *)malloc((size_t)nelmts * sizeof(nds_list_t));
        if (ndsbuf_w == NULL)
            goto out;
    }
    else {
        dsbuf = (ds_list_t *)malloc((size_t)nelmts * sizeof(ds_list_t));
        if (dsbuf == NULL)
            goto out;
        if (H5Aread(aid, ntid, dsbuf) < 0)
            goto out;
        dsbuf_w = (ds_list_t *)malloc((size_t)nelmts * sizeof(ds_list_t));
        if (dsbuf_w == NULL)
            goto out;
    }
    
    for (i = 0; i < nelmts; i++) {
        if (is_new_ref) {
            ndsbuf_w[i].dim_idx = ndsbuf[i].dim_idx;
            tmp_id              = H5Ropen_object(&ndsbuf[i].ref, H5P_DEFAULT, H5P_DEFAULT);
            if (tmp_id < 0)
                goto out;
            if (H5Rcreate_object(tmp_id, ".", H5P_DEFAULT, &ndsbuf_w[i].ref) < 0) {
                H5Dclose(tmp_id);
                goto out;
            }
            H5Dclose(tmp_id);
        }
        else {
            dsbuf_w[i] = dsbuf[i];
        }
    }
    for (ii = 0; ii < nelmts; ii++) {
        
        if (is_new_ref) {
            tmp_idx = ndsbuf_w[ii].dim_idx;
        }
        else {
            tmp_idx = dsbuf_w[ii].dim_idx;
        }
        if (idx == tmp_idx) {
            
            if (is_new_ref) {
                
                nref = ndsbuf_w[ii].ref;
                if ((did_i = H5Ropen_object(&nref, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                    goto out;
            }
            else {
                
                ref = dsbuf_w[ii].ref;
                if ((did_i = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref)) < 0)
                    goto out;
            }

            
            if (H5Oget_info3(did_i, &tmp_oi, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (H5Dclose(did_i) < 0)
                goto out;

            
            if (did_oi.fileno == tmp_oi.fileno) {
                int token_cmp;

                if (H5Otoken_cmp(did, &did_oi.token, &tmp_oi.token, &token_cmp) < 0)
                    goto out;
                if (!token_cmp) {
                    
                    if (is_new_ref) {
                        ndsbuf_w[ii] = ndsbuf_w[nelmts - 1];
                    }
                    else {
                        dsbuf_w[ii] = dsbuf_w[nelmts - 1];
                    }
                    nelmts--;
                    found_dset = 1;
                    break;
                } 
            }     
        }         
    }             

    
    if (H5Aclose(aid) < 0)
        goto out;

    

    
    if (found_dset == 0)
        goto out;

    

    
    if (H5Adelete(dsid, REFERENCE_LIST) < 0)
        goto out;

    
    if (nelmts) {
        
        dims[0] = (hsize_t)nelmts;

        if ((sid_w = H5Screate_simple(1, dims, NULL)) < 0)
            goto out;

        
        if ((aid = H5Acreate2(dsid, REFERENCE_LIST, tid, sid_w, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        if (is_new_ref) {
            if (H5Awrite(aid, ntid, ndsbuf_w) < 0)
                goto out;
        }
        else {
            if (H5Awrite(aid, ntid, dsbuf_w) < 0)
                goto out;
        }

        if (H5Aclose(aid) < 0)
            goto out;
    } 

    
    if (is_new_ref) {
        if (H5Treclaim(tid, sid, H5P_DEFAULT, ndsbuf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (sid_w > 0) {
            if (H5Treclaim(tid, sid_w, H5P_DEFAULT, ndsbuf_w) < 0)
                goto out;
            if (H5Sclose(sid_w) < 0)
                goto out;
        }
    }
    else {
        if (H5Treclaim(tid, sid, H5P_DEFAULT, dsbuf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (sid_w > 0) {
            if (H5Treclaim(tid, sid_w, H5P_DEFAULT, dsbuf_w) < 0)
                goto out;
            if (H5Sclose(sid_w) < 0)
                goto out;
        }
    }
    
    if (H5Tclose(tid) < 0)
        goto out;
    if (H5Tclose(ntid) < 0)
        goto out;
    if (is_new_ref) {
        free(ndsbuf);
        free(ndsbuf_w);
        ndsbuf   = NULL;
        ndsbuf_w = NULL;
    }
    else {
        free(dsbuf);
        free(dsbuf_w);
        dsbuf   = NULL;
        dsbuf_w = NULL;
    }

    return SUCCEED;

    
out:
    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(ntid);
        H5Tclose(tid);

        if (ndsbuf) {
            free(ndsbuf);
            ndsbuf = NULL;
        }
        if (ndsbuf_w) {
            free(ndsbuf_w);
            ndsbuf_w = NULL;
        }
        if (dsbuf) {
            free(dsbuf);
            dsbuf = NULL;
        }
        if (buf) {
            
            for (i = 0; i < rank; i++) {
                if (buf[i].p)
                    free(buf[i].p);
            }
            free(buf);
            buf = NULL;
        }
    }
    H5E_END_TRY
    return FAIL;
}

htri_t
H5DSis_attached(hid_t did, hid_t dsid, unsigned int idx)
{
    htri_t      has_dimlist;
    htri_t      has_reflist;
    hssize_t    nelmts;
    hid_t       sid;                    
    hid_t       tid  = H5I_INVALID_HID; 
    hid_t       ntid = H5I_INVALID_HID; 
    hid_t       aid  = H5I_INVALID_HID; 
    int         rank;                   
    nds_list_t *ndsbuf = NULL;          
    ds_list_t  *dsbuf  = NULL;          
    H5R_ref_t   nref;                   
    hobj_ref_t  ref;                    
    hvl_t      *buf = NULL;             
    hid_t       dsid_j;                 
    hid_t       did_i;                  
    H5O_info2_t oi1, oi2, oi3, oi4;
    H5I_type_t  it1, it2;
    int         i;
    int         found_dset = 0, found_ds = 0;
    htri_t      is_scale;
    bool        is_new_ref;

    

    if ((is_scale = H5DSis_scale(did)) < 0)
        return FAIL;

    
    if (is_scale == 1)
        return FAIL;

    
    if (H5Oget_info3(did, &oi1, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (H5Oget_info3(dsid, &oi2, H5O_INFO_BASIC) < 0)
        return FAIL;

    
    if (oi1.fileno == oi2.fileno) {
        int token_cmp;

        if (H5Otoken_cmp(did, &oi1.token, &oi2.token, &token_cmp) < 0)
            return FAIL;
        if (!token_cmp)
            return FAIL;
    } 

    

    if (H5DSwith_new_ref(did, &is_new_ref) < 0)
        return FAIL;

    
    if ((it1 = H5Iget_type(did)) < 0)
        return FAIL;
    if ((it2 = H5Iget_type(dsid)) < 0)
        return FAIL;

    if (H5I_DATASET != it1 || H5I_DATASET != it2)
        return FAIL;

    

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (idx > ((unsigned)rank - 1))
        return FAIL;

    
    if ((has_dimlist = H5Aexists(did, DIMENSION_LIST)) < 0)
        return FAIL;

    

    if (has_dimlist > 0) {
        if ((aid = H5Aopen(did, DIMENSION_LIST, H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        
        buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));
        if (buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, buf) < 0)
            goto out;

        
        for (i = 0; i < (int)buf[idx].len; i++) {
            if (is_new_ref) {
                
                nref = ((H5R_ref_t *)buf[idx].p)[i];

                
                if ((dsid_j = H5Ropen_object(&nref, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                    goto out;
            }
            else {
                
                ref = ((hobj_ref_t *)buf[idx].p)[i];

                
                if ((dsid_j = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref)) < 0)
                    goto out;
            }

            
            if (H5Oget_info3(dsid, &oi1, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (H5Oget_info3(dsid_j, &oi2, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (oi1.fileno == oi2.fileno) {
                int token_cmp;

                if (H5Otoken_cmp(did, &oi1.token, &oi2.token, &token_cmp) < 0)
                    goto out;
                if (!token_cmp)
                    found_ds = 1;
            } 

            
            if (H5Dclose(dsid_j) < 0)
                goto out;
        }

        
        if (H5Treclaim(tid, sid, H5P_DEFAULT, buf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(buf);
        buf = NULL;
    } 

    

    
    if ((has_reflist = H5Aexists(dsid, REFERENCE_LIST)) < 0)
        goto out;

    

    if (has_reflist > 0) {
        if ((aid = H5Aopen(dsid, REFERENCE_LIST, H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        
        if ((ntid = H5Tget_native_type(tid, H5T_DIR_ASCEND)) < 0)
            goto out;

        
        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        if ((nelmts = H5Sget_simple_extent_npoints(sid)) < 0)
            goto out;

        if (is_new_ref) {
            ndsbuf = (nds_list_t *)malloc((size_t)nelmts * sizeof(nds_list_t));
            if (ndsbuf == NULL)
                goto out;
            if (H5Aread(aid, ntid, ndsbuf) < 0)
                goto out;
        }
        else {
            dsbuf = (ds_list_t *)malloc((size_t)nelmts * sizeof(ds_list_t));
            if (dsbuf == NULL)
                goto out;
            if (H5Aread(aid, ntid, dsbuf) < 0)
                goto out;
        }

        

        for (i = 0; i < nelmts; i++) {

            if (is_new_ref) {
                nref = ndsbuf[i].ref;
                
                if ((did_i = H5Ropen_object(&nref, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                    goto out;
            }
            else {
                ref = dsbuf[i].ref;
                
                if ((did_i = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref)) < 0)
                    goto out;
            }

            
            if (H5Oget_info3(did, &oi3, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (H5Oget_info3(did_i, &oi4, H5O_INFO_BASIC) < 0)
                goto out;

            
            if (oi3.fileno == oi4.fileno) {
                int token_cmp;

                if (H5Otoken_cmp(did, &oi3.token, &oi4.token, &token_cmp) < 0)
                    goto out;
                if (is_new_ref) {
                    if (!token_cmp && (idx == ndsbuf[i].dim_idx))
                        found_dset = 1;
                }
                else {
                    if (!token_cmp && (idx == dsbuf[i].dim_idx))
                        found_dset = 1;
                }
            } 

            
            if (H5Dclose(did_i) < 0)
                goto out;
        } 

        
        if (is_new_ref) {
            if (H5Treclaim(ntid, sid, H5P_DEFAULT, ndsbuf) < 0)
                goto out;
        }
        else {
            if (H5Treclaim(ntid, sid, H5P_DEFAULT, dsbuf) < 0)
                goto out;
        }
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(ntid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;

        if (ndsbuf) {
            free(ndsbuf);
            ndsbuf = NULL;
        }
        if (dsbuf) {
            free(dsbuf);
            dsbuf = NULL;
        }
    } 

    if (found_ds && found_dset)
        return 1;
    else
        return 0;

    
out:
    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(tid);
        H5Tclose(ntid);
    }
    H5E_END_TRY

    if (buf) {
        free(buf);
        buf = NULL;
    }
    if (ndsbuf) {
        free(ndsbuf);
        ndsbuf = NULL;
    }
    if (dsbuf) {
        free(dsbuf);
        dsbuf = NULL;
    }
    return FAIL;
}

herr_t
H5DSiterate_scales(hid_t did, unsigned int dim, int *ds_idx, H5DS_iterate_t visitor, void *visitor_data)
{
    hid_t      scale_id;
    int        rank;
    H5R_ref_t  nref;                  
    hobj_ref_t ref;                   
    hid_t      sid;                   
    hid_t      tid = H5I_INVALID_HID; 
    hid_t      aid = H5I_INVALID_HID; 
    hvl_t     *buf = NULL;            
    H5I_type_t it;                    
    herr_t     ret_value = 0;
    int        j_idx;
    int        nscales;
    htri_t     has_dimlist;
    int        i;
    bool       is_new_ref;

    
    
    if ((it = H5Iget_type(did)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    

    if (H5DSwith_new_ref(did, &is_new_ref) < 0)
        return FAIL;

    
    if ((nscales = H5DSget_num_scales(did, dim)) < 0)
        return FAIL;

    
    if (ds_idx != NULL) {
        if (*ds_idx >= nscales)
            return FAIL;
    }

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    if (dim >= (unsigned)rank)
        return FAIL;

    
    if ((has_dimlist = H5Aexists(did, DIMENSION_LIST)) < 0)
        return FAIL;
    if (has_dimlist == 0)
        return SUCCEED;

    else if (has_dimlist > 0) {
        if ((aid = H5Aopen(did, DIMENSION_LIST, H5P_DEFAULT)) < 0)
            goto out;
        if ((tid = H5Aget_type(aid)) < 0)
            goto out;
        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        
        buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));

        if (buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, buf) < 0)
            goto out;

        if (buf[dim].len > 0) {
            if (ds_idx != NULL)
                j_idx = *ds_idx;
            else
                j_idx = 0;

            
            for (i = j_idx; i < nscales; i++) {
                if (is_new_ref) {
                    
                    nref = ((H5R_ref_t *)buf[dim].p)[i];

                    
                    H5E_BEGIN_TRY
                    {
                        
                        if ((scale_id = H5Ropen_object(&nref, H5P_DEFAULT, H5P_DEFAULT)) < 0)
                            goto out;
                    }
                    H5E_END_TRY
                }
                else {
                    
                    ref = ((hobj_ref_t *)buf[dim].p)[i];

                    
                    H5E_BEGIN_TRY
                    {
                        
                        if ((scale_id = H5Rdereference2(did, H5P_DEFAULT, H5R_OBJECT, &ref)) < 0)
                            goto out;
                    }
                    H5E_END_TRY
                }

                
                if (ds_idx != NULL) {
                    *ds_idx = i;
                }

                if ((ret_value = (visitor)(did, dim, scale_id, visitor_data)) != 0) {
                    

                    
                    if (H5Dclose(scale_id) < 0)
                        goto out;

                    break;
                }

                
                if (H5Dclose(scale_id) < 0)
                    goto out;

            } 
        }     

        
        if (H5Treclaim(tid, sid, H5P_DEFAULT, buf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;

        free(buf);
        buf = NULL;
    } 

    return ret_value;

out:
    H5E_BEGIN_TRY
    {
        if (buf) {
            H5Treclaim(tid, sid, H5P_DEFAULT, buf);
            free(buf);
        }
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(tid);
    }
    H5E_END_TRY

    return FAIL;
}

herr_t
H5DSset_label(hid_t did, unsigned int idx, const char *label)
{
    htri_t       has_labels;
    hid_t        sid = H5I_INVALID_HID; 
    hid_t        tid = H5I_INVALID_HID; 
    hid_t        aid = H5I_INVALID_HID; 
    int          rank;                  
    hsize_t      dims[1];               
    H5I_type_t   it;                    
    unsigned int i;
    union {                     
        char       **buf;       
        char const **const_buf; 
    } u;

    memset(&u, 0, sizeof(u));

    
    
    if ((it = H5Iget_type(did)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    if (label == NULL)
        return FAIL;

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    if (idx >= (unsigned)rank)
        return FAIL;

    

    
    if ((has_labels = H5Aexists(did, DIMENSION_LABELS)) < 0)
        return FAIL;

    

    if (has_labels == 0) {
        dims[0] = (hsize_t)rank;

        
        if ((sid = H5Screate_simple(1, dims, NULL)) < 0)
            goto out;

        
        if ((tid = H5Tcopy(H5T_C_S1)) < 0)
            goto out;
        if (H5Tset_size(tid, H5T_VARIABLE) < 0)
            goto out;

        
        if ((aid = H5Acreate2(did, DIMENSION_LABELS, tid, sid, H5P_DEFAULT, H5P_DEFAULT)) < 0)
            goto out;

        
        u.const_buf = (char const **)malloc((size_t)rank * sizeof(char *));

        if (u.const_buf == NULL)
            goto out;

        for (i = 0; i < (unsigned int)rank; i++)
            u.const_buf[i] = NULL;

        
        u.const_buf[idx] = label;

        
        if (H5Awrite(aid, tid, u.const_buf) < 0)
            goto out;

        
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        if (u.const_buf) {
            free(u.const_buf);
            u.const_buf = NULL;
        }
    }

    

    else {

        if ((aid = H5Aopen(did, DIMENSION_LABELS, H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        
        u.buf = (char **)malloc((size_t)rank * sizeof(char *));

        if (u.buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, (void *)u.buf) < 0)
            goto out;

        
        if (u.buf[idx])
            free(u.buf[idx]);

        
        u.const_buf[idx] = label;

        
        if (H5Awrite(aid, tid, u.buf) < 0)
            goto out;

        
        u.buf[idx] = NULL;

        
        for (i = 0; i < (unsigned int)rank; i++) {
            if (u.buf[i])
                free(u.buf[i]);
        }

        
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        if (u.buf) {
            free(u.buf);
            u.buf = NULL;
        }
    }

    return SUCCEED;

    

out:
    if (u.buf) {
        if (u.buf[idx])        
            u.buf[idx] = NULL; 
        
        for (i = 0; i < (unsigned int)rank; i++) {
            if (u.buf[i])
                free(u.buf[i]);
        }
        free(u.buf);
    }
    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(tid);
    }
    H5E_END_TRY
    return FAIL;
}

ssize_t
H5DSget_label(hid_t did, unsigned int idx, char *label, size_t size)
{
    htri_t     has_labels;
    hid_t      sid = H5I_INVALID_HID; 
    hid_t      tid = H5I_INVALID_HID; 
    hid_t      aid = H5I_INVALID_HID; 
    int        rank;                  
    char     **buf = NULL;            
    H5I_type_t it;                    
    size_t     nbytes = 0;
    size_t     copy_len;
    int        i;

    
    
    if ((it = H5Iget_type(did)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    if (idx >= (unsigned)rank)
        return FAIL;

    

    
    if ((has_labels = H5Aexists(did, DIMENSION_LABELS)) < 0)
        return FAIL;

    
    if (has_labels == 0) {
        if (label)
            label[0] = 0;
        return 0;
    }

    

    if ((aid = H5Aopen(did, DIMENSION_LABELS, H5P_DEFAULT)) < 0)
        goto out;

    if ((tid = H5Aget_type(aid)) < 0)
        goto out;

    
    buf = (char **)malloc((size_t)rank * sizeof(char *));

    if (buf == NULL)
        goto out;

    
    if (H5Aread(aid, tid, buf) < 0)
        goto out;

    
    if (buf[idx] != NULL) {
        
        nbytes = strlen(buf[idx]);

        
        copy_len = MIN(size - 1, nbytes);

        
        if (label) {
            memcpy(label, buf[idx], copy_len);

            
            label[copy_len] = '\0';
        }
    }
    
    for (i = 0; i < rank; i++) {
        if (buf[i])
            free(buf[i]);
    }

    
    if (H5Tclose(tid) < 0)
        goto out;
    if (H5Aclose(aid) < 0)
        goto out;
    if (buf) {
        free(buf);
        buf = NULL;
    }

    return (ssize_t)nbytes;

    
out:
    if (buf) {
        
        for (i = 0; i < rank; i++) {
            if (buf[i])
                free(buf[i]);
        }
        free(buf);
    }
    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(tid);
    }
    H5E_END_TRY
    return FAIL;
}

ssize_t
H5DSget_scale_name(hid_t did, char *name, size_t size)
{
    hid_t      aid = H5I_INVALID_HID; 
    hid_t      tid = H5I_INVALID_HID; 
    hid_t      sid = H5I_INVALID_HID; 
    H5I_type_t it;                    
    size_t     nbytes;
    size_t     copy_len;
    htri_t     has_name;
    char      *buf = NULL;

    
    
    if ((it = H5Iget_type(did)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    if ((H5DSis_scale(did)) <= 0)
        return FAIL;

    

    
    if ((has_name = H5Aexists(did, "NAME")) < 0)
        return FAIL;
    if (has_name == 0)
        return 0;

    

    if ((aid = H5Aopen(did, "NAME", H5P_DEFAULT)) < 0)
        return FAIL;

    
    if ((sid = H5Aget_space(aid)) < 0)
        goto out;

    
    if ((tid = H5Aget_type(aid)) < 0)
        goto out;

    
    if ((nbytes = H5Tget_size(tid)) == 0)
        goto out;

    
    buf = (char *)malloc(nbytes * sizeof(char));
    if (buf == NULL)
        goto out;

    
    if (H5Aread(aid, tid, buf) < 0)
        goto out;

    
    copy_len = MIN(size - 1, nbytes);

    
    if (name) {
        memcpy(name, buf, copy_len);

        
        name[copy_len] = '\0';
    }

    
    if (H5Tclose(tid) < 0)
        goto out;
    if (H5Aclose(aid) < 0)
        goto out;
    if (H5Sclose(sid) < 0)
        goto out;
    if (buf)
        free(buf);

    return (ssize_t)(nbytes - 1);

    
out:
    H5E_BEGIN_TRY
    {
        H5Aclose(aid);
        H5Tclose(tid);
        H5Sclose(sid);
    }
    H5E_END_TRY
    if (buf)
        free(buf);
    return FAIL;
}

htri_t
H5DSis_scale(hid_t did)
{
    hid_t       tid = H5I_INVALID_HID; 
    hid_t       aid = H5I_INVALID_HID; 
    htri_t      attr_class;            
    htri_t      is_ds = -1;            
    H5I_type_t  it;                    
    char       *buf = NULL;            
    size_t      string_size;           
    H5T_class_t type_class;
    H5T_str_t   strpad;

    
    
    if ((it = H5Iget_type(did)) < 0)
        goto out;

    if (H5I_DATASET != it)
        goto out;

    
    if ((attr_class = H5Aexists(did, "CLASS")) < 0)
        goto out;

    if (attr_class == 0) {
        is_ds = 0;
        goto out;
    }
    else {
        if ((aid = H5Aopen(did, "CLASS", H5P_DEFAULT)) < 0)
            goto out;

        if ((tid = H5Aget_type(aid)) < 0)
            goto out;

        
        if ((type_class = H5Tget_class(tid)) < 0)
            goto out;
        if (H5T_STRING != type_class) {
            is_ds = 0;
            goto out;
        }
        
        if ((strpad = H5Tget_strpad(tid)) < 0)
            goto out;
        if (H5T_STR_NULLTERM != strpad) {
            is_ds = 0;
            goto out;
        }

        
        if ((string_size = H5Tget_size(tid)) == 0)
            goto out;
        if (string_size != 16) {
            is_ds = 0;
            goto out;
        }

        buf = (char *)malloc((size_t)string_size * sizeof(char));
        if (buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, buf) < 0)
            goto out;

        
        if (strncmp(buf, DIMENSION_SCALE_CLASS, MIN(strlen(DIMENSION_SCALE_CLASS), strlen(buf))) == 0)
            is_ds = 1;

        free(buf);

        if (H5Tclose(tid) < 0)
            goto out;

        if (H5Aclose(aid) < 0)
            goto out;
    }
out:
    if (is_ds < 0) {
        free(buf);
        H5E_BEGIN_TRY
        {
            H5Aclose(aid);
            H5Tclose(tid);
        }
        H5E_END_TRY
    }
    return is_ds;
}

int
H5DSget_num_scales(hid_t did, unsigned int idx)
{
    htri_t     has_dimlist;
    hid_t      sid;                   
    hid_t      tid = H5I_INVALID_HID; 
    hid_t      aid = H5I_INVALID_HID; 
    int        rank;                  
    hvl_t     *buf = NULL;            
    H5I_type_t it;                    
    int        nscales;

    
    
    if ((it = H5Iget_type(did)) < 0)
        return FAIL;

    if (H5I_DATASET != it)
        return FAIL;

    
    
    if ((sid = H5Dget_space(did)) < 0)
        return FAIL;

    
    if ((rank = H5Sget_simple_extent_ndims(sid)) < 0)
        goto out;

    
    if (H5Sclose(sid) < 0)
        goto out;

    
    if (idx >= (unsigned int)rank)
        return FAIL;

    
    if ((has_dimlist = H5Aexists(did, DIMENSION_LIST)) < 0)
        return FAIL;

    
    if (has_dimlist == 0)
        return 0;

    
    else {
        if ((aid = H5Aopen(did, DIMENSION_LIST, H5P_DEFAULT)) < 0)
            goto out;
        if ((tid = H5Aget_type(aid)) < 0)
            goto out;
        if ((sid = H5Aget_space(aid)) < 0)
            goto out;

        
        buf = (hvl_t *)malloc((size_t)rank * sizeof(hvl_t));
        if (buf == NULL)
            goto out;

        
        if (H5Aread(aid, tid, buf) < 0)
            goto out;

        nscales = (int)buf[idx].len;

        
        if (H5Treclaim(tid, sid, H5P_DEFAULT, buf) < 0)
            goto out;
        if (H5Sclose(sid) < 0)
            goto out;
        if (H5Tclose(tid) < 0)
            goto out;
        if (H5Aclose(aid) < 0)
            goto out;
        free(buf);
        buf = NULL;
    } 

    return nscales;

    
out:
    H5E_BEGIN_TRY
    {
        H5Sclose(sid);
        H5Aclose(aid);
        H5Tclose(tid);
    }
    H5E_END_TRY

    if (buf)
        free(buf);

    return FAIL;
}

static herr_t
H5DS_is_reserved(hid_t did, bool *is_reserved)
{
    htri_t has_class;
    hid_t  tid = H5I_INVALID_HID;
    hid_t  aid = H5I_INVALID_HID;
    char  *buf = NULL;  
    size_t string_size; 

    
    if ((has_class = H5Aexists(did, "CLASS")) < 0)
        return FAIL;
    if (has_class == 0) {
        *is_reserved = false;
        return SUCCEED;
    }

    if ((aid = H5Aopen(did, "CLASS", H5P_DEFAULT)) < 0)
        goto error;
    if ((tid = H5Aget_type(aid)) < 0)
        goto error;

    
    if (H5T_STRING != H5Tget_class(tid))
        goto error;

    
    if (H5T_STR_NULLTERM != H5Tget_strpad(tid))
        goto error;

    
    if ((string_size = H5Tget_size(tid)) == 0)
        goto error;
    if (NULL == (buf = malloc(string_size * sizeof(char))))
        goto error;

    
    if (H5Aread(aid, tid, buf) < 0)
        goto error;

    if (strncmp(buf, IMAGE_CLASS, MIN(strlen(IMAGE_CLASS), strlen(buf))) == 0 ||
        strncmp(buf, PALETTE_CLASS, MIN(strlen(PALETTE_CLASS), strlen(buf))) == 0 ||
        strncmp(buf, TABLE_CLASS, MIN(strlen(TABLE_CLASS), strlen(buf))) == 0)
        *is_reserved = true;
    else
        *is_reserved = false;

    free(buf);

    if (H5Tclose(tid) < 0)
        goto error;
    if (H5Aclose(aid) < 0)
        goto error;

    return SUCCEED;

error:
    H5E_BEGIN_TRY
    {
        H5Tclose(tid);
        H5Aclose(aid);
    }
    H5E_END_TRY

    free(buf);

    return FAIL;
}
