#' @title Generic Conservation Test Framework
#' @description Perform a generic conservation test by comparing observed TXI patterns
#' against a null distribution generated by permutation.
#' 
#' @param phyex_set A PhyloExpressionSet object
#' @param test_name Character string naming the test
#' @param scoring_function Function to compute test statistic from TXI profile
#' @param fitting_dist Distribution object for fitting the null distribution
#' @param alternative Character string specifying alternative hypothesis: 
#' "two-sided", "greater", or "less" (default: "two-sided")
#' @param p_label Label for p-value in results (default: p_label)
#' @param custom_null_txis Optional custom null TXI distribution (default: NULL)
#' @param plot_result Logical indicating whether to plot results (default: TRUE)
#' 
#' @return A ConservationTestResult object containing test statistics and p-values
#' 
#' @details
#' This function provides a generic framework for conservation testing by:
#' 1. Generating null TXI distributions via permutation
#' 2. Computing test statistics using the provided scoring function
#' 3. Fitting the specified distribution to the null sample
#' 4. Computing p-values based on the alternative hypothesis
#' 
#' 
#' @seealso \code{\link{stat_flatline_test}}, \code{\link{stat_early_conservation_test}}
stat_generic_conservation_test <- function(phyex_set,
                                           test_name,
                                           scoring_function,
                                           fitting_dist,
                                           alternative = c("two-sided", "greater", "less"),
                                           p_label=p_label,
                                           custom_null_txis = NULL,
                                           plot_result=TRUE
                                           ) {
    # check arguments
    S7::check_is_S7(phyex_set, PhyloExpressionSetBase)
    stopifnot(is.function(scoring_function))
    alternative <- match.arg(alternative)
    
    # 1. Simulate the TXI distribution under null hypothesis, via permutation
    if (!is.null(custom_null_txis))
        null_txis <- custom_null_txis
    else
        null_txis <- phyex_set@null_conservation_txis
    
    # 2. Compute the null distribution, using the appropriate scoring function
    null_sample <- apply(null_txis, 1, scoring_function)
    
    # 3. Compute test statistic using the same scoring function
    test_stat <- scoring_function(phyex_set@TXI)
    
    # 4. Estimate null distribution parameters
    params <- fitting_dist@fitting_function(null_sample)
    
    res <- ConservationTestResult(method_name=test_name,
                                  test_stat=test_stat,
                                  fitting_dist=fitting_dist,
                                  params=params,
                                  alternative=alternative,
                                  null_sample=null_sample,
                                  data_name=phyex_set@name,
                                  null_txis=null_txis,
                                  test_txi=phyex_set@TXI,
                                  p_label=p_label
                                  )
    if (plot_result)
        print(plot(res))
    
   return(res)
}

#' @title Diagnose Test Robustness
#' @description Evaluate the robustness of conservation tests across different sample sizes
#' for null distribution generation.
#' 
#' @param test Function representing the conservation test to evaluate
#' @param phyex_set A PhyloExpressionSet object
#' @param sample_sizes Numeric vector of sample sizes to test (default: c(500, 1000, 5000, 10000))
#' @param plot_result Logical indicating whether to plot results (default: TRUE)
#' @param num_reps Number of replicates for each sample size (default: 5)
#' @param ... Additional arguments passed to the test function
#' 
#' @return A data frame with test results across different sample sizes
#' 
#' @details
#' This function assesses how consistent test results are across different sample sizes
#' for null distribution generation, helping to determine appropriate sample sizes
#' for reliable testing.
#' 
#' @examples
#' # Diagnose flatline test robustness
#' robustness <- diagnose_test_robustness(stat_flatline_test, 
#'                                        example_phyex_set,
#'                                        sample_sizes=c(10,20),
#'                                        plot_result=FALSE,
#'                                        num_reps=3)
#' 
#' @import ggplot2
#' 
#' @export
diagnose_test_robustness <- function(test, 
                                     phyex_set,
                                     sample_sizes=c(500, 1000, 5000, 10000),
                                     plot_result=TRUE,
                                     num_reps=5,
                                     ...) {

    f <- function(size) {
        null_txis <- stat_generate_conservation_txis(phyex_set@strata,
                                                     phyex_set@expression_collapsed,
                                                     sample_size=size)
        return(test(phyex_set, custom_null_txis=null_txis, ..., plot_result=FALSE))
    }
    
    res_vec <- purrr::map(rep(sample_sizes, each=num_reps), f) |>
        purrr::map_dbl(~ .x@p_value)
    
    df <- data.frame(pval=res_vec, sample_size=rep(sample_sizes, each=num_reps))
    
    p <- ggplot(df, aes(x=factor(sample_size), y=pval, color="Independent run")) +
        geom_jitter(height=0, width=0.02) +
        scale_x_discrete(breaks=sample_sizes) +
        scale_y_continuous(transform='log10') +
        labs(x="Null sample size", 
             y="p value",
             color="Legend",
             title="Robustness of test p-values against different null distribution sample sizes") +
        scale_color_manual(values="black") +
        theme_minimal()
    
    return(p)
}
