% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_diamond.R
\name{geom_diamond}
\alias{geom_diamond}
\title{Geom for drawing diamond plots of lattice structures}
\usage{
geom_diamond(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  na.rm = FALSE,
  linetype = "solid",
  linewidth = NA,
  nudge_x = 0,
  nudge_y = 0.125,
  show.legend = NA,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{mapping}{Aesthetic mappings, usually created with \code{\link[ggplot2:aes]{ggplot2::aes()}}.}

\item{data}{A data frame representing the lattice structure to plot.}

\item{stat}{Statistical transformation to apply; defaults to \code{"identity"}.}

\item{position}{Position adjustment for the geom; defaults to \code{"identity"}.}

\item{na.rm}{Logical; if \code{TRUE}, missing values are silently removed.}

\item{linetype}{Line type for edges; defaults to \code{"solid"}.}

\item{linewidth}{Width of edges connecting parent and child nodes. If set to
\code{NA}, edge widths are determined by the \code{linewidth} aesthetic. If no
aesthetic is provided, a default width of \code{0.5} is used.}

\item{nudge_x}{Horizontal nudge applied to labels.}

\item{nudge_y}{Vertical nudge applied to labels.}

\item{show.legend}{Logical; whether to include a legend. Defaults to \code{FALSE}.}

\item{inherit.aes}{Logical; whether to inherit aesthetics from the plot.
Defaults to \code{TRUE}.}

\item{...}{Additional arguments passed to \code{\link[ggplot2:layer]{ggplot2::layer()}}.}
}
\value{
A \code{ggplot2} layer object that adds a diamond lattice visualization
to an existing plot.
}
\description{
Create a custom \code{ggplot2} geom for visualizing lattice structures as
\emph{diamond plots}. This geom is particularly useful for displaying
association rules and their ancestor–descendant relationships in a clear,
compact graphical form.
}
\details{
In a diamond plot, nodes (diamonds) represent items or conditions within
the lattice, while edges denote inclusion (subset) relationships between
them. The geom combines node and edge rendering with flexible control over
aesthetics such as labels, color, and size.

\strong{Concept overview}

A \emph{lattice} represents inclusion relationships between conditions. Each
node corresponds to a condition, and a line connects a condition to its
direct descendants:

\if{html}{\out{<div class="sourceCode">}}\preformatted{       \{a\}          <- ancestor (parent)
      /   \\
  \{a,b\}   \{a,c\}     <- direct descendants (children)
     \\     /
     \{a,b,c\}        <- leaf condition
}\if{html}{\out{</div>}}

The layout positions broader (more general) conditions above their
descendants. This helps visualize hierarchical structures such as those
produced by association rule mining or subset lattices.

\strong{Supported aesthetics}
\itemize{
\item \code{condition} – character vector of conditions formatted with
\code{\link[=format_condition]{format_condition()}}. Each condition defines one node in the lattice.
The hierarchy is determined by subset inclusion: a condition \eqn{X}
is a descendant of \eqn{Y} if \eqn{Y \subset X}. Each condition must
be unique.
\item \code{label} – optional text label for each node. If omitted,
the condition string is used.
\item \code{colour} – border color of the node.
\item \code{fill} – interior color of the node.
\item \code{size} – size of nodes.
\item \code{shape} – node shape.
\item \code{alpha} – transparency of nodes.
\item \code{stroke} – border line width of nodes.
\item \code{linewidth} – edge width between parent and child nodes,
computed as the difference of this aesthetic between them.
}
}
\examples{
\dontrun{
library(ggplot2)

# Prepare data by partitioning numeric columns into fuzzy or crisp sets
part <- partition(iris, .breaks = 3)

# Find all antecedents with "Sepal" for rules with consequent "Species=setosa"
rules <- dig_associations(part,
                          antecedent = starts_with("Sepal"),
                          consequent = `Species=setosa`,
                          min_length = 0,
                          max_length = Inf,
                          min_coverage = 0,
                          min_support = 0,
                          min_confidence = 0,
                          measures = c("lift", "conviction"),
                          max_results = Inf)

# Add abbreviated labels for readability
rules$abbrev <- shorten_condition(rules$antecedent)

# Plot the lattice of rules as a diamond diagram
ggplot(rules) +
  aes(condition = antecedent,
      fill = confidence,
      linewidth = confidence,
      size = coverage,
      label = abbrev) +
  geom_diamond()
}

}
\author{
Michal Burda
}
