#' @title
#' Extracting the incomparability set of a poset element.
#'
#' @description
#' Extracts the elements incomparable with the input element, in the poset.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .

#' @param element A character string (the names of a single poset element).
#'
#' @return
#' A vector of character strings (the names of the poset elements incomparable with the input element).
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)

#' incmp <- IncomparabilitySetOf(pos, "a")

#'
#' @name IncomparabilitySetOf
#' @export IncomparabilitySetOf
IncomparabilitySetOf <- function(poset, element) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    if (!is.character(element))
      stop("element must be a string")
    if (length(element) != 1)
      stop("element must be a single value")
    result <- .Call("_IncomparabilitySetOf", poset@ptr, element)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
