# base color palette
.base.colors <- c("blue","red","darkgreen","magenta","orange","cyan","gold","black")

#' Bivariate PCA-style Scatter Plot
#'
#' This function generates a 2D scatter plot with support for multiple groups,
#' labels, arrows from the origin, reference circles, cross axes, and full style customization using `ggplot2`.
#'
#' @param points.list List of numeric objects (matrices, data.frames, or lists of vectors), each representing a group of 2D points.
#' @param style.points List of `ggplot2` styles applied per group. If `NULL`, `geom_point()` is used by default.
#' @param style.circle List of styles for the reference circle (passed to `ggforce::geom_circle`).
#' @param radius.circle Radius (or vector of radii) to draw circles centered at the origin. Default value `1`, if `0`, no circles are drawn.
#' @param labels `"auto"` generates numeric labels by group, or it can be a vector/list of custom labels.
#' @param labels.style List of styles for the labels, passed to `ggrepel::geom_text_repel`.
#' @param draw.labels Logical. If `TRUE`, labels are drawn on the points.
#' @param vars.direction Directions of projected variables.
#' @param style.vars Style for projected variables.
#' @param radius.vars Radii used to scale variable arrows.
#' @param join.dots Logical or list. If `TRUE`, connects points by group. If a list, connects points as manually defined.
#' @param style.join List of styles for connecting points (passed to `geom_path()`).
#' @param base.colors Vector of base colors used for the groups.
#' @param axes Logical. If `FALSE`, all axis elements are removed.
#' @param frame Logical. Not directly used; may be reserved for future use.
#' @param hide.ticks Logical. If `TRUE`, hides axis ticks and text.
#' @param proportion Fixed aspect ratio of the plot (to avoid distortion).
#' @param xlim X-axis limits.
#' @param ylim Y-axis limits.
#' @param axes.xy Logical. If `TRUE`, draws cross axes (X = 0, Y = 0) with defined style.
#' @param style.axes.xy List of styles for the XY cross axes (e.g., `linetype`, `color`, etc.).
#' @param arrows.points Logical. If `TRUE`, draws arrows from the origin to each point.
#' @param factor.arrow Factor to shorten the arrows.
#' @param style.arrows List of styles for the arrows.
#'
#' @return A `ggplot` object with the generated plot.
#'
#' @import ggplot2
#' @import ggforce
#' @import ggrepel
#' @importFrom ggplot2 ggtitle
#' @importFrom grid arrow unit
#'
#' @seealso \code{\link{statis.dual}}
#'
#' @examples
#' data(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98)
#' labels <- c("95","96","97","98")
#'
#' res <- statis.dual(list(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98), labels.tables = labels)
#'
#' # Interstructure
#' t <- ggplot2::ggtitle("Interstructure")
#' plot.statis.dual.circle(points.list = list(res$interstructure), labels = res$labels.tables) + t
#'
#' # Circle of correlations (all variables)
#' t <- ggplot2::ggtitle("Correlation (all variables)")
#' plot.statis.dual.circle(list(res$supervariables), labels = row.names(res$supervariables)) + t
#'
#' # Circle of correlations (variables selected)
#' selected.variables <- c("Ph", "Temp", "DBO", "ST", "PO4", "NO3", "POD", "Cal")
#' superv.sel.df <- select.super.variables(res$supervariables, res$vars.names, selected.variables)
#'
#' t <- ggplot2::ggtitle("Correlation (selected variables)")
#' plot.statis.dual.circle(list(superv.sel.df), labels = row.names(superv.sel.df)) + t
#'
#' @rawNamespace export(plot.statis.dual.circle)
plot.statis.dual.circle <- function(points.list,
                                    style.points = list(list(size = 3)),
                                    style.circle = list(), radius.circle = 1,
                                    labels = "auto", labels.style = NULL, draw.labels = TRUE,
                                    vars.direction = NULL, style.vars = list(), radius.vars = c(0.5, 1.0),
                                    join.dots = FALSE, style.join = list(),
                                    base.colors = .base.colors,
                                    axes = TRUE, frame = TRUE, hide.ticks = TRUE,
                                    proportion = 1, xlim = NULL, ylim = NULL,
                                    axes.xy = TRUE,
                                    style.axes.xy = list(linewidth = 0.35, linetype = "dashed", color = "gray40"),
                                    arrows.points = TRUE, factor.arrow = 0.95,
                                    style.arrows = list(color = "red", linewidth = 0.6, arrow = grid::arrow(length = grid::unit(0.2, "cm")))) {

  colores_auto <- function(k, base) rep(base, length.out = k)

  # Converts each element of points.list into a standardized data frame
  # Ensures columns are x, y and adds group and index identifiers
  a_df <- function(g, gid){
    if (is.list(g) && !is.data.frame(g) && !is.matrix(g)) g <- do.call(rbind, g)
    g <- as.data.frame(g); stopifnot(ncol(g) >= 2)
    colnames(g)[1:2] <- c("x","y") # Force first two columns to be x,y coordinates
    g$group <- factor(gid) # group = ID of dataset inside points.list
    g$idx <- seq_len(nrow(g)); g # idx = index inside each group, used for automatic labeling
  }

  # --------------------------------------------------------------------
  # 1. Convert the list of point sets into one combined data frame
  # --------------------------------------------------------------------
  groups <- seq_along(points.list)
  df <- do.call(rbind, Map(a_df, points.list, groups))
  cols <- colores_auto(length(points.list), base.colors) # Generate final vector of colors (one per group)

  # --------------------------------------------------------------------
  # 2. Handle labels
  #    - "auto" = numeric sequence per group
  #    - or custom labels supplied by user
  # --------------------------------------------------------------------
  if (identical(labels, "auto")) {
    labs <- ave(df$idx, df$group, FUN = function(i) as.character(seq_along(i)))
  } else {
    labs <- if (is.list(labels)) unlist(labels) else labels
    stopifnot(length(labs) == nrow(df))
  }
  df$label <- labs

  # --------------------------------------------------------------------
  # 3. Compute shortened arrow ends
  # Prevents arrows from reaching full length for readability
  # --------------------------------------------------------------------
  df$xfin <- df$x * factor.arrow
  df$yfin <- df$y * factor.arrow

  # --------------------------------------------------------------------
  # 4. Initialize ggplot object
  # --------------------------------------------------------------------
  p <- ggplot(df, aes(x = x, y = y, color = group))

  # --------------------------------------------------------------------
  # 5. Draw principal axes (horizontal and vertical lines at origin)
  # --------------------------------------------------------------------
  if (isTRUE(axes.xy)) {
    sty <- style.axes.xy
    p <- p + do.call(geom_hline, c(list(yintercept = 0), sty)) +
      do.call(geom_vline, c(list(xintercept = 0), sty))
  }
  # --------------------------------------------------------------------
  # 6. Draw arrows from origin toward each observation/variable
  # --------------------------------------------------------------------
  if (isTRUE(arrows.points)) {
    p <- p + do.call(geom_segment,
                     c(list(data = df,
                            mapping = aes(x = 0, y = 0, xend = xfin, yend = yfin, color = group),
                            inherit.aes = FALSE), style.arrows))
  }
  # --------------------------------------------------------------------
  # 7. Draw points (each group can have its own style)
  # If style.points = NULL, use default geom_point()
  # --------------------------------------------------------------------
  if (is.null(style.points)) {
    p <- p + geom_point()
  } else {
    # Loop through each group to apply its own style
    for (g in levels(df$group)) {
      i <- as.integer(g); dfg <- df[df$group == g, , drop = FALSE]
      sty <- style.points[[i]]; if (is.null(sty)) sty <- list()
      p <- p + do.call(geom_point,
                       c(list(data = dfg, mapping = aes(x = x, y = y, color = group)), sty))
    }
  }
  # --------------------------------------------------------------------
  # 8. Optionally join dots with paths (trajectories)
  # Used for STATIS trajectories or temporal profiles.
  # --------------------------------------------------------------------
  if (isTRUE(join.dots)) {
    p <- p + geom_path(aes(group = group))
  } else if (is.list(join.dots) && length(join.dots) > 0) {
    for (jl in join.dots) {
      jldf <- as.data.frame(do.call(rbind, jl)); colnames(jldf)[1:2] <- c("x","y")
      p <- p + do.call(geom_path,
                       c(list(data = jldf, mapping = aes(x = x, y = y), inherit.aes = FALSE),
                         style.join))
    }
  }
  # --------------------------------------------------------------------
  # 9. Draw correlation circle(s)
  # radius.circle = 0 → skip
  # --------------------------------------------------------------------
  if (!(length(radius.circle) == 1 && radius.circle == 0)) {
    radios <- if (length(radius.circle) == 1) c(radius.circle) else radius.circle
    circ_df <- do.call(rbind, lapply(radios, function(r) data.frame(x0 = 0, y0 = 0, r = r)))
    cs <- utils::modifyList(list(color = "black", fill = NA), style.circle)
    p <- p + do.call(ggforce::geom_circle,
                     c(list(data = circ_df, mapping = aes(x0 = x0, y0 = y0, r = r),
                            inherit.aes = FALSE), cs))
  }
  # --------------------------------------------------------------------
  # 10. Draw labels using ggrepel (avoids overlap)
  # --------------------------------------------------------------------
  if (isTRUE(draw.labels)) {
    lbl_default <- list(size = 3, segment.alpha = 0.4, max.overlaps = Inf, box.padding = 0.25)
    sty <- if (is.null(labels.style)) lbl_default else utils::modifyList(lbl_default, labels.style)
    p <- p + do.call(ggrepel::geom_text_repel,
                     c(list(data = df,
                            mapping = aes(x = x, y = y, label = label, color = group),
                            inherit.aes = FALSE), sty))
  }
  # --------------------------------------------------------------------
  # 11. Remove frame, axes, ticks depending on user settings
  # --------------------------------------------------------------------
  if (!axes) {
    p <- p + theme(axis.line  = element_blank(),
                   axis.text  = element_blank(),
                   axis.ticks = element_blank(),
                   axis.title = element_blank())
  }
  if (hide.ticks) {
    p <- p + theme(axis.text = element_blank(),
                   axis.ticks = element_blank())
  }
  # --------------------------------------------------------------------
  # 12. Apply proportion, limits, colors and minimal theme
  # coord_fixed keeps aspect ratio = proportion
  # --------------------------------------------------------------------
  p + coord_fixed(ratio = proportion, xlim = xlim, ylim = ylim, clip = "on") +
    scale_color_manual(values = cols, guide = "none") +
    theme_minimal()
}

#' Plot Variable Trajectories in STATIS DUAL
#'
#' Visualizes the evolution of one or more variables across the different tables
#' in a STATIS DUAL analysis. Each trajectory represents the sequence of positions
#' of a variable in the compromise space.
#'
#' @param vars Vector of variable names to plot (must match the names in \code{trajectories}).
#' @param trajectories List generated by \code{statis.dual()$trajectories}, where each element is a K x 2 matrix.
#' @param labels.tables Vector of length K with the names or labels of the tables.
#' @param .range List with axis limits: \code{list(x = c(xmin, xmax), y = c(ymin, ymax))}. If \code{NULL}, limits are computed automatically.
#' @param style.line List with line style for the trajectories.
#' @param point.size Size of the points at each position along the trajectory.
#' @param base.colors Vector of base colors to distinguish the variables.
#'
#' @return A \code{ggplot} object showing the trajectories of the selected variables.
#'
#' @seealso \code{\link{plot.statis.dual.circle}}, \code{\link{statis.dual}}
#'
#' @examples
#' data(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98)
#' labels = c("95","96","97","98")
#'
#' res <- statis.dual(list(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98), labels.tables = labels)
#'
#' # If you want to select some variables
#' vars.A <- c("Ph","ST","NO3")
#' t <- ggplot2::ggtitle(sprintf("Trayectorias (%s)", paste(vars.A, collapse = ", ")))
#'
#' plot.statis.dual.trajectories(vars.A, res$trajectories, res$labels.tables) + t
#'
#' # If you want to select an specific variable
#' vars.1 <- "Temp"
#' t <- ggplot2::ggtitle(sprintf("Trajectory (%s)", vars.1))
#'
#' plot.statis.dual.trajectories(vars.1, res$trajectories, res$labels.tables) + t
#'
#' # All variables
#' t <- ggplot2::ggtitle("Trajectories (all variables)")
#' plot.statis.dual.trajectories(res$vars.names, res$trajectories, res$labels.tables) + t
#'
#' @rawNamespace export(plot.statis.dual.trajectories)
plot.statis.dual.trajectories <- function(vars,
                                          trajectories,
                                          labels.tables,
                                          .range = NULL,
                                          style.line = list(linetype=2, linewidth=0.5, color="orange"),
                                          point.size = 3,
                                          base.colors = c("red","blue","brown","darkgreen","purple",
                                                          "orange","cyan4","gold3","black")) {

  # ----------------------------------------------------------------------
  # 1. Input validation: ensure selected variables exist in trajectories
  # ----------------------------------------------------------------------
  stopifnot(is.character(vars), all(vars %in% names(trajectories)))

  # ----------------------------------------------------------------------
  # 2. Extract trajectories for each variable
  #    Each trajectory is a K × 2 matrix (one point per table)
  #    Convert each trajectory to a data frame with columns x,y
  Lpts <- lapply(vars, function(nm){
    M <- trajectories[[nm]]; df <- as.data.frame(M); colnames(df) <- c("x","y"); df
  })

  # ----------------------------------------------------------------------
  # 3. Build labels for each trajectory
  #    First point gets: "VariableName Table1"
  #    Remaining points get only table labels
  # ----------------------------------------------------------------------
  labels.list <- lapply(vars, function(nm) c(paste(nm, labels.tables[1]), labels.tables[-1]))

  # ----------------------------------------------------------------------
  # 4. Styling of labels and points
  # ----------------------------------------------------------------------
  styles.labels <- replicate(length(Lpts), list(size=3), simplify = FALSE)
  styles.points <- replicate(length(Lpts), list(size=point.size), simplify = FALSE)
  col.vec <- rep(base.colors, length.out = length(Lpts)) # Colors for each trajectory

  # ----------------------------------------------------------------------
  # 5. Ensures all trajectories are properly visible in the plot
  # ----------------------------------------------------------------------
  if (is.null(.range)) {
    allxy <- do.call(rbind, Lpts)
    xr <- range(allxy$x); yr <- range(allxy$y)
    pad <- 0.05 * max(diff(xr), diff(yr))
    .range <- list(x = xr + c(-pad, pad), y = yr + c(-pad, pad))
  }

  # ----------------------------------------------------------------------
  # 6. Build the plot using plot.statis.dual.circle
  # ----------------------------------------------------------------------
  plot.statis.dual.circle(
    points.list = Lpts,
    radius.circle = 0,             # No circles in trajectory plots
    labels = labels.list,
    labels.style = styles.labels,
    style.points = styles.points,
    join.dots = TRUE,              # Connect dots to form the trajectory
    style.join = style.line,
    base.colors = col.vec,
    axes = TRUE, frame = TRUE, hide.ticks = FALSE,
    proportion = 1,
    xlim = .range$x, ylim = .range$y,
    arrows.points = FALSE          # No arrows, only points connected by lines
  )
}
