#' @title STATIS Method
#'
#' @description
#' Applies the STATIS method to a set of matrices (data tables) with the same rows.
#' Is a multivariate analysis technique that allows studying the common structure
#' and the evolution of individuals and variables across multiple tables.
#'
#' @param matrices List of numeric matrices (at least 2), all with the same number of rows (individuals).
#' @param selected.tables Select a subset of tables. If `NULL`, all tables are included.
#' @param selected.rows Select a subset of rows. If `NULL`, all rows are included.
#' @param table.labels Optional vector with names for the tables. It must have the same length as the number of tables.
#'
#' @return A list with the following elements:
#' \item{n}{Number of rows (individuals).}
#' \item{r}{Number of tables.}
#' \item{p}{Vector with the number of columns per table.}
#' \item{S}{List of centered matrices.}
#' \item{W}{List of proximity matrices.}
#' \item{X}{Matrix of interstructure (vectorization of W).}
#' \item{acp.inter}{PCA results of the interstructure: eigenvalues, eigenvectors, components, correlations.}
#' \item{XT}{Weighted average of the matrices.}
#' \item{acp.intra}{PCA results of the average: eigenvalues, eigenvectors, components, correlations.}
#' \item{IND}{Concatenated matrix with all W stacked (individual evolution).}
#' \item{Omega}{Projection of individual evolution onto the principal components.}
#' \item{circle.inter}{Data to plot the correlation circle between tables.}
#' \item{circle.intra}{Data to plot the variable evolution circle.}
#' \item{plane.individuals}{Data to plot the average individuals plane.}
#' \item{plane.evolution}{Data to plot the evolution of the individuals.}
#'
#' @seealso \code{\link{plot.statis.circle}}, \code{\link{plot.statis.plane}}
#'
#' @examples
#' data(expert1, expert2, expert3)
#'
#' labels <- c("Expert 1", "Expert 2", "Expert 3")
#'
#' # If you want to select an specific table or tables
#' res <- statis(list(expert1, expert2, expert3), selected.tables = c(1, 3), table.labels = labels)
#'
#' # If you want to select an specific row or rows
#' res <- statis(list(expert1, expert2, expert3), selected.rows = c(1, 5), table.labels = labels)
#'
#' # If you want to select some tables and rows at the same time
#' res <- statis(list(expert1,expert2,expert3), selected.tables=c(1, 3), selected.rows=c(1, 4), labels)
#'
#' # All tables and rows selected
#' res <- statis(list(expert1, expert2, expert3), table.labels = labels)
#'
#' # How to use res
#' inter <- res$circle.inter
#' plot.statis.circle(inter$points, inter$inertia, inter$labels, inter$title)
#'
#' evolution <- res$plane.evolution
#' plot.statis.plane(evolution$points, evolution$inertia, evolution$labels, evolution$title)
#'
#' @export
statis <- function(matrices, selected.tables = NULL, selected.rows = NULL, table.labels = NULL) {
  # ----------------------------------------------------------------------
  # Internal helper to normalize selection of indexes.
  # Accepts:
  #   - NULL      → select all
  #   - logical   → TRUE/FALSE mask
  #   - numeric   → either 0/1 mask or integer indexes (1..m)
  .norm_idx <- function(sel, m) {
    if (is.null(sel)) return(seq_len(m))
    if (is.logical(sel)) {
      if (length(sel) != m) stop("Logical selection must have length ", m, ".")
      idx <- which(sel)
      if (!length(idx)) stop("No items selected.")
      return(idx)
    }
    if (is.numeric(sel)) {
      if (length(sel) == m && all(sel %in% c(0,1))) {
        idx <- which(sel == 1)
        if (!length(idx)) stop("No items selected.")
        return(idx)
      } else {
        if (!all(sel %in% seq_len(m))) {
          stop("Numeric selection must be indexed in 1..", m,
               " or mask 0/1 of length m or indexed 1..m.")
        }
        return(as.integer(sel))
      }
    }
    stop("Selection must be NULL, logical (length m) or numerical (0/1 of length m or indexes 1..m).")
  }

  # ----------------------------------------------------------------------
  # Initialization of variables
  stopifnot(is.list(matrices), length(matrices) >= 2)
  r <- length(matrices)
  n <- nrow(matrices[[1]])
  if (any(sapply(matrices, nrow) != n)) stop("All matrices must have the same number of rows (n).")

  `%||%` <- function(a, b) if (is.null(a)) b else a

  # ----------------------------------------------------------------------
  # >>> NEW: Parameterizable labels for tables <<<
  #   - If not provided: Table1, Table2, ..., TableR
  #   - If provided: must match number of tables
  if (is.null(table.labels)) {
    tablabs <- paste0("Table", seq_len(r))
  } else {
    if (length(table.labels) != r) {
      stop("`table.labels` must have a length equal to the number of tables (", r, ").")
    }
    tablabs <- as.character(table.labels)
  }

  # ----------------------------------------------------------------------
  # INDIVIDUAL LABELS (row labels)
  # Keep labels only if they are consistent across all matrices
  # Otherwise use 1..n
  rowlabs0 <- rownames(matrices[[1]])
  if (!is.null(rowlabs0) && all(vapply(matrices, function(M) identical(rownames(M), rowlabs0), logical(1)))) {
    rowlabs <- rowlabs0
  } else {
    rowlabs <- as.character(seq_len(n))
  }

  # ----------------------------------------------------------------------
  # STEP 1 — INITIALIZATION
  # Get all p_k, number of variables in each table
  p <- sapply(matrices, ncol)

  # Generate the S matrix with the centered data tables
  S <- lapply(matrices, center)

  # ----------------------------------------------------------------------
  # STEP 2 — INTERSTRUCTURE (Steps 2 to 5):
  # Calculate correlation/covariance matrices W_i
  W <- lapply(S, function(Si) Si %*% t(Si))

  # ----------------------------------------------------------------------
  # STEP 3 — Build X:
  #   - Each Wi is vectorized and used as a column of X
  #   - Allows PCA to measure similarity between tables
  X <- do.call(cbind, lapply(W, function(Wi) as.vector(Wi)))

  # ----------------------------------------------------------------------
  # STEP 4 — PCA OF THE INTERSTRUCTURE (PCA on X)
  res1 <- acp(X, reduce = TRUE)
  U1 <- res1$U; V1 <- res1$V; C1 <- res1$C; CP1 <- res1$CP
  lambda <- U1[1] # dominant eigenvalue
  u <- V1[, 1]# first eigenvector (global compromise weights)

  lambda1 <- U1[1]
  lambda2 <- U1[2]
  inertia.inter <- round(100 * (lambda1+lambda2) / sum(U1), 2) # Percentage of explained inertia by first two dimensions

  # ----------------------------------------------------------------------
  # STEP 5 — SAVE CORRELATION CIRCLE OF TABLES
  idx_tab <- .norm_idx(selected.tables, r)  # table selection (same as before)
  circle.inter <- NULL
  if (ncol(CP1) >= 2) {
    PTS1 <- CP1[idx_tab, 1:2, drop = FALSE]
    circle.inter <- list(
      points    = as.matrix(PTS1),
      labels = as.character(tablabs[idx_tab]),
      inertia   = inertia.inter,
      title    = "Correlation Circle of the tables"
    )
  }

  # ----------------------------------------------------------------------
  # STEP 6 — INTRASTRUCTURE (Steps 7 to 17): Study of variable evolution across tables

  # STEP 7: Evolution of variables (Steps 8 to 13)

  # STEP 8: Calculate Beta
  Beta <- as.numeric(u) / sqrt(lambda)

  # ----------------------------------------------------------------------
  # STEP 9: Calculate the matrix XT with column names
  # XT is the input for the intrastructure PCA
  XT <- do.call(cbind, lapply(seq_len(r), function(i) {
    Xi <- S[[i]] * u[i]
    cn <- colnames(matrices[[i]])
    if (is.null(cn)) cn <- paste0("V", seq_len(ncol(matrices[[i]])))
    bad <- is.na(cn) | cn == ""
    if (any(bad)) cn[bad] <- paste0("V", which(bad))
    colnames(Xi) <- paste0(tablabs[i], ":", cn)
    Xi
  }))

  # ----------------------------------------------------------------------
  # STEP 10: Centered and reduced PCA of XT
  res2 <- acp(XT, reduce = TRUE)
  U2 <- res2$U; V2 <- res2$V; C2 <- res2$C; CP2 <- res2$CP
  lambda1 <- U2[1]
  lambda2 <- U2[2]
  inertia.intra <- round(100 * (lambda1+lambda2) / sum(U2), 2)

  # ----------------------------------------------------------------------
  # STEP 11: Save matrix C with the first principal components
  k.keep <- min(n, ncol(C2))
  C <- C2[, 1:k.keep, drop = FALSE]

  # ----------------------------------------------------------------------
  # STEP 12: Save data to graph the correlation circle for the evolution of the variables
  circle.intra <- NULL
  if (ncol(CP2) >= 2) {
    points2 <- CP2[, 1:2, drop = FALSE]
    labs2   <- colnames(XT)

    # filter by selected tables (prefixes)
    prefijos.keep <- paste0(tablabs[idx_tab], ":")
    keep.mask <- Reduce(`|`, lapply(prefijos.keep, function(pr) startsWith(labs2, pr)))
    points2 <- points2[keep.mask, , drop = FALSE]
    labs2   <- labs2[keep.mask]

    circle.intra <- list(
      points    = as.matrix(points2),
      labels = labs2,
      inertia   = inertia.intra,
      title    = "Correlation Circle of the evolution of the variables"
    )
  }

  # ----------------------------------------------------------------------
  # STEP 13 — MAIN PLANE OF AVERAGE INDIVIDUALS
  plane.individuals <- NULL
  if (ncol(C) >= 2) {
    # Row selection (individuals)
    idx.row <- .norm_idx(selected.rows, n)
    pts.ind <- C[idx.row, 1:2, drop = FALSE]
    plane.individuals <- list(
      points    = as.matrix(pts.ind),
      labels = as.character(rowlabs[idx.row]),
      inertia   = inertia.intra,
      title    = "Principal Plane of Average Individuals"
    )
  }

  # ----------------------------------------------------------------------
  # STEPS 14–17 — EVOLUTION OF INDIVIDUALS
  # STEP 15: Calculate the IND matrix
  IND <- do.call(rbind, W)
  evo.labs <- unlist(lapply(seq_len(r), function(i) paste0(rowlabs, "\u00B7", tablabs[i])), use.names = FALSE)

  # ----------------------------------------------------------------------
  # STEP 16: Calculate the coordinates of the evolution of individuals (Omega)
  Omega <- IND %*% C

  # ----------------------------------------------------------------------
  # STEP 17: Save data for the plane with the evolution of individuals
  plane.evolution <- NULL
  if (ncol(Omega) >= 2) {
    # Apply the same row selection to each table block
    idx.row <- .norm_idx(selected.rows, n)
    # Build indexes on r blocks of size n
    idx_evo <- as.vector(unlist(lapply(seq_len(r), function(i) (i - 1L) * n + idx.row)))
    pts.ind.evo <- Omega[idx_evo, 1:2, drop = FALSE]
    plane.evolution <- list(
      points    = as.matrix(pts.ind.evo),
      labels = as.character(evo.labs[idx_evo]),
      inertia   = inertia.intra,
      title    = "Principal Plane of the Evolution of Individuals"
    )
  }

  # ----------------------------------------------------------------------
  # FINAL OUTPUT: returns everything required for analysis and plotting
  list(
    n = n, r = r, p = p,
    S = S, W = W,
    X = X,
    acp.inter = list(U = U1, V = V1, C = C1, CP = CP1, lambda1 = lambda, u1 = u, Beta = Beta),
    XT = XT,
    acp.intra = list(U = U2, V = V2, C = C2, CP = CP2, C.keep = C),
    IND = IND,
    Omega = Omega,
    # Data to graph
    circle.inter    = circle.inter,
    circle.intra    = circle.intra,
    plane.individuals = plane.individuals,
    plane.evolution  = plane.evolution
  )
}
