% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design_table.R
\name{design_table}
\alias{design_table}
\title{Create Design Comparison Table for Two Co-Primary Endpoints}
\usage{
design_table(
  param_grid,
  rho_values = c(0, 0.3, 0.5, 0.8),
  r = 1,
  alpha = 0.025,
  beta = 0.2,
  endpoint_type = c("continuous", "binary", "mixed_cont_binary", "mixed_count_cont"),
  Test = "AN",
  known_var = TRUE,
  nMC = 1000,
  output_var = NULL
)
}
\arguments{
\item{param_grid}{A data.frame containing parameter combinations. Required
columns depend on endpoint_type and calculation_mode:
\itemize{
\item For continuous endpoints (sample size): delta1, delta2, sd1, sd2
\item For continuous endpoints (power): n1, n2, delta1, delta2, sd1, sd2
\item For binary endpoints (sample size): p11, p12, p21, p22
\item For binary endpoints (power): n1, n2, p11, p12, p21, p22
\item For mixed continuous-binary (sample size): delta, sd, p1, p2
\item For mixed continuous-binary (power): n1, n2, delta, sd, p1, p2
\item For mixed count-continuous (sample size): r1, r2, nu, t, mu1, mu2, sd
\item For mixed count-continuous (power): n1, n2, r1, r2, nu, t, mu1, mu2, sd
}}

\item{rho_values}{Numeric vector of correlation values to evaluate. Default
is c(0, 0.3, 0.5, 0.8).}

\item{r}{Allocation ratio (n1/n2). Required for sample size calculation.
Default is 1.}

\item{alpha}{One-sided significance level. Default is 0.025.}

\item{beta}{Type II error rate (1 - power). Required for sample size
calculation. Default is 0.2 (power = 0.8).}

\item{endpoint_type}{Character string specifying endpoint type:
"continuous", "binary", "mixed_cont_binary", or "mixed_count_cont".}

\item{Test}{Test method for binary endpoints: "AN" (asymptotic normal),
"ANc" (with continuity correction), "AS" (arcsine), or "ASc". Default is
"AN". Only used for binary and mixed_cont_binary endpoints.}

\item{known_var}{Logical indicating whether variance is known for continuous
endpoints. Default is TRUE.}

\item{nMC}{Number of Monte Carlo simulations for certain calculations.
Default is 1000.}

\item{output_var}{Character string specifying which variable to output in
the result columns: "N" (total sample size, default for sample size
calculation) or "powerCoprimary" (co-primary power, default for power
calculation).}
}
\value{
A data.frame of class "twoCoprimary_table" with:
\itemize{
\item Parameter columns (from param_grid)
\item Result columns for each correlation value (rho_0.0, rho_0.3, etc.)
}
}
\description{
Generates a comprehensive table comparing sample sizes or power across
different parameter combinations and correlation values. This function is
useful for sensitivity analyses and exploring how design parameters affect
statistical properties.
}
\details{
This function performs systematic calculations across all combinations of
parameters specified in param_grid and correlation values in rho_values.

The calculation mode (sample size vs power) is automatically determined:
\itemize{
\item If param_grid contains n1 and n2: calculates power
\item Otherwise: calculates sample size (requires r, alpha, beta)
}

For binary endpoints with two correlations (rho1, rho2), both are set to
the same value from rho_values for each calculation.

The output format follows the style of Sozu et al. (2011), with parameters
displayed in the leftmost columns and results for each correlation in
subsequent columns.
}
\examples{
# Sample size calculation for continuous endpoints
param_grid <- expand.grid(
  delta1 = c(0.3, 0.5),
  delta2 = c(0.1, 0.2, 0.3),
  sd1 = c(1.0, 1.5),
  sd2 = c(1.0, 1.5)
)

result <- design_table(
  param_grid = param_grid,
  rho_values = c(0, 0.3, 0.5, 0.8),
  r = 1,
  alpha = 0.025,
  beta = 0.2,
  endpoint_type = "continuous"
)
print(result)

# Power calculation for continuous endpoints
param_grid_power <- expand.grid(
  n1 = c(50, 100),
  n2 = c(50, 100),
  delta1 = 0.5,
  delta2 = 0.5,
  sd1 = 1.0,
  sd2 = 1.0
)

result_power <- design_table(
  param_grid = param_grid_power,
  rho_values = c(0, 0.3, 0.5, 0.8),
  alpha = 0.025,
  endpoint_type = "continuous"
)
print(result_power)

# Binary endpoints
param_grid_binary <- expand.grid(
  p11 = c(0.6, 0.7),
  p12 = c(0.4, 0.5),
  p21 = c(0.4, 0.5),
  p22 = c(0.2, 0.3)
)

result_binary <- design_table(
  param_grid = param_grid_binary,
  rho_values = c(0.3, 0.5, 0.7),
  r = 1,
  alpha = 0.025,
  beta = 0.2,
  endpoint_type = "binary",
  Test = "AN"
)
print(result_binary)

}
\references{
Sozu, T., Kanou, T., Hamada, C., & Yoshimura, I. (2011). Power and sample
size calculations in clinical trials with multiple primary variables.
Japanese Journal of Biometrics, 27, 83-96.
}
