#' Plot of Molecular Mass (M) vs. Number of Carbon Atoms (C)
#'
#' @title Plot of Molecular Mass (M) vs. Number of Carbon Atoms (C)
#' @name uplot_cvm
#' @family plots
#'
#' @description
#' Generates a scatter plot of *molecular mass (M)* versus *carbon atom count (C)*,
#' color-coded by a selected variable (`z_var`).
#'
#' This visualization follows the concept of the Carbon-vs-Mass (CvM) diagram
#' introduced by **Reemtsma (2010)**.
#'
#' @inheritParams main_docu
#' @inheritDotParams f_colorz
#'
#' @param df A data.table containing columns:
#'   \describe{
#'     \item{`nm`}{Molecular mass (Da)}
#'     \item{`12C`}{Number of carbon atoms}
#'     \item{`z_var`}{Variable used for color projection}
#'   }
#' @param z_var Character. Column used for color mapping.
#' @param palname Character. Palette name passed to `f_colorz()`.
#'
#' @return A ggplot2 or Plotly object.
#'
#' @references
#' Reemtsma, T. (2010).
#' *The carbon versus mass diagram to visualize and exploit FTICR-MS data of natural organic matter.*
#' J. Mass Spectrom., **45**, 382–390. \doi{10.1002/jms.1722}
#'
#' @examples
#' uplot_cvm(mf_data_demo, z_var = "co_tot", logo = FALSE)
#'
#' @export
uplot_cvm <- function(
    df,
    z_var = "co_tot",
    palname = "redblue",
    tf = FALSE,
    col_bar = TRUE,
    gg_size = 12,
    logo = TRUE,
    plotly = FALSE,
    ...
) {

  df <- data.table::as.data.table(df)

  zz <- NULL

  # --- Validation --------------------------------------------------------------
  if (!"nm" %in% names(df))
    stop("Column 'nm' (molecular mass) is missing.")

  if (!"12C" %in% names(df))
    stop("Column '12C' (carbon count) is missing.")

  if (!z_var %in% names(df))
    stop("Column '", z_var, "' is missing.")

  # --- Aggregate to unique (nm, C) combinations -------------------------------
  df_CvM <- df[, .(zz = max(get(z_var))), by = .(nm, `12C`)]

  # optional log-transform
  if (isTRUE(tf)) {
    if (any(df_CvM$zz <= 0))
      stop("Log transform requires z_var > 0.")
    df_CvM[, zz := log10(zz)]
  }

  # --- Build palette using f_colorz() -----------------------------------------
  pal_vec <- f_colorz(
    seq(0, 1, length.out = 256),
    palname = palname,
    return_palette = TRUE,
    ...
  )

  # --- Base ggplot -------------------------------------------------------------
  p <- ggplot2::ggplot(df_CvM, aes(x = nm, y = `12C`, color = zz)) +
    geom_point(size = 2) +
    scale_color_gradientn(
      colours = pal_vec,
      name = z_var,
      guide = if (col_bar) "colourbar" else "none"
    ) +
    labs(
      x = "Molecular mass (Da)",
      y = "Number of carbon atoms",
      title = "Carbon vs Mass (CvM) Diagram",
      caption = if (logo && !plotly) "UltraMassExplorer" else NULL
    ) +
    theme_uplots(base_size = gg_size)

  # --- Plotly version ----------------------------------------------------------
  if (isTRUE(plotly)) {

    pp <- plotly::ggplotly(p)

    if (logo) {
      pp <- pp |>
        plotly::layout(
          annotations = list(
            list(
              text = "<i>UltraMassExplorer</i>",
              xref = "paper", yref = "paper",
              x = 1, y = -0.12,
              xanchor = "right", yanchor = "top",
              showarrow = FALSE,
              font = list(size = gg_size, color = "gray40")
            )
          )
        )
    }

    return(pp)
  }

  return(p)
}
