% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_BKP.R, R/predict_DKP.R
\name{predict}
\alias{predict}
\alias{predict.BKP}
\alias{predict.DKP}
\title{Predict Method for BKP or DKP Models}
\usage{
\method{predict}{BKP}(object, Xnew, CI_level = 0.95, threshold = 0.5, ...)

\method{predict}{DKP}(object, Xnew, CI_level = 0.95, ...)
}
\arguments{
\item{object}{An object of class \code{"BKP"} or \code{"DKP"}, typically
returned by \code{\link{fit.BKP}} or \code{\link{fit.DKP}}.}

\item{Xnew}{A numeric matrix (or vector) of new input locations where
predictions are desired.}

\item{CI_level}{Credible level for prediction intervals (default is
\code{0.95}, corresponding to 95\% CI).}

\item{threshold}{Classification threshold for binary prediction based on
posterior mean (used only for BKP; default is \code{0.5}).}

\item{...}{Additional arguments passed to generic \code{predict} methods
(currently not used; included for S3 method consistency).}
}
\value{
A list with the following components:
\describe{
\item{\code{Xnew}}{The new input locations.}
\item{\code{mean}}{BKP: Posterior mean of the success probability at each location.
DKP: A matrix of posterior mean class probabilities (rows = inputs, columns = classes).}
\item{\code{variance}}{BKP: Posterior variance of the success probability.
DKP: A matrix of posterior variances for each class.}
\item{\code{lower}}{BKP: Lower bound of the prediction interval (e.g., 2.5th percentile for 95\% CI).
DKP: A matrix of lower bounds for each class (e.g., 2.5th percentile).}
\item{\code{upper}}{BKP: Upper bound of the prediction interval (e.g., 97.5th percentile for 95\% CI).
DKP: A matrix of upper bounds for each class (e.g., 97.5th percentile).}
\item{\code{class}}{BKP: Predicted binary label (0 or 1), based on posterior mean and threshold, if \code{m = 1}.
DKP: Predicted class label (i.e., the class with the highest posterior mean probability).}
}
}
\description{
Generates posterior predictive summaries from a fitted BKP or
DKP model at new input locations.
}
\examples{
# ============================================================== #
# ========================= BKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2,2), nrow=1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model1 <- fit.BKP(X, y, m, Xbounds=Xbounds)

# Prediction
Xnew = matrix(seq(-2, 2, length = 10), ncol=1) #new data points
predict(model1, Xnew)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define 2D latent function and probability transformation
true_pi_fun <- function(X) {
  if(is.null(nrow(X))) X <- matrix(X, nrow=1)
  m <- 8.6928
  s <- 2.4269
  x1 <- 4*X[,1]- 2
  x2 <- 4*X[,2]- 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19- 14*x1 + 3*x1^2- 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1- 3*x2)^2 *
    (18- 32*x1 + 12*x1^2 + 48*x2- 36*x1*x2 + 27*x2^2)
  f <- log(a*b)
  f <- (f- m)/s
  return(pnorm(f))  # Transform to probability
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model2 <- fit.BKP(X, y, m, Xbounds=Xbounds)

# Prediction
x1 <- seq(Xbounds[1,1], Xbounds[1,2], length.out = 10)
x2 <- seq(Xbounds[2,1], Xbounds[2,2], length.out = 10)
Xnew <- expand.grid(x1 = x1, x2 = x2)
predict(model2, Xnew)

# ============================================================== #
# ========================= DKP Examples ======================= #
# ============================================================== #

#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model1 <- fit.DKP(X, Y, Xbounds = Xbounds)

# Prediction
Xnew = matrix(seq(-2, 2, length = 10), ncol=1) #new data points
predict(model1, Xnew)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define latent function and transform to 3-class probabilities
true_pi_fun <- function(X) {
  if (is.null(nrow(X))) X <- matrix(X, nrow = 1)
  m <- 8.6928; s <- 2.4269
  x1 <- 4 * X[,1] - 2
  x2 <- 4 * X[,2] - 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1 - 3*x2)^2 *
    (18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2)
  f <- (log(a * b) - m) / s
  p <- pnorm(f)
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model2 <- fit.DKP(X, Y, Xbounds = Xbounds)
print(model2)

# Prediction
x1 <- seq(Xbounds[1,1], Xbounds[1,2], length.out = 10)
x2 <- seq(Xbounds[2,1], Xbounds[2,2], length.out = 10)
Xnew <- expand.grid(x1 = x1, x2 = x2)
predict(model2, Xnew)

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\seealso{
\code{\link{fit.BKP}} for fitting Beta Kernel Process models.
\code{\link{fit.DKP}} for fitting Dirichlet Kernel Process models.
\code{\link{plot.BKP}} for visualizing fitted BKP models.
\code{\link{plot.DKP}} for visualizing fitted DKP models.
}
\keyword{BKP}
\keyword{DKP}
