% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quantile_BKP.R, R/quantile_DKP.R
\name{quantile}
\alias{quantile}
\alias{quantile.BKP}
\alias{quantile.DKP}
\title{Posterior Quantiles from a Fitted BKP or DKP Model}
\usage{
\method{quantile}{BKP}(x, probs = c(0.025, 0.5, 0.975), ...)

\method{quantile}{DKP}(x, probs = c(0.025, 0.5, 0.975), ...)
}
\arguments{
\item{x}{An object of class \code{BKP} or \code{DKP}, typically the result of
a call to \code{\link{fit_BKP}} or \code{\link{fit_DKP}}.}

\item{probs}{Numeric vector of probabilities specifying which posterior
quantiles to return. Defaults to \code{c(0.025, 0.5, 0.975)}.}

\item{...}{Additional arguments (currently unused).}
}
\value{
For \code{BKP}: a numeric vector (if \code{length(probs) = 1}) or a
numeric matrix (if \code{length(probs) > 1}) of posterior quantiles. Rows
correspond to observations, and columns correspond to the requested
probabilities.

For \code{DKP}: a numeric matrix (if \code{length(probs) = 1}) or a 3D
array (if \code{length(probs) > 1}) of posterior quantiles. Dimensions
correspond to observations × classes × probabilities.
}
\description{
Compute posterior quantiles from a fitted \code{BKP} or
\code{DKP} model. For a \code{BKP} object, this returns the posterior
quantiles of the positive class probability. For a \code{DKP} object, this
returns posterior quantiles for each class probability.
}
\details{
For a \code{BKP} model, posterior quantiles are computed from the
Beta Kernel Process for the positive class probability. For a \code{DKP}
model, posterior quantiles for each class are approximated using the Beta
approximation of the marginal distributions of the posterior Dirichlet
distribution.
}
\examples{
# -------------------------- BKP ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model <- fit_BKP(X, y, m, Xbounds = Xbounds)

# Extract posterior quantiles
quantile(model)

# -------------------------- DKP ---------------------------
#' set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p1 <- 1/(1+exp(-3*X))
  p2 <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model <- fit_DKP(X, Y, Xbounds = Xbounds)

# Extract posterior quantiles
quantile(model)

}
\seealso{
\code{\link{fit_BKP}}, \code{\link{fit_DKP}} for model fitting.
}
\keyword{BKP}
\keyword{DKP}
