#' Run pyDarwin Model Search
#'
#' This function runs a pyDarwin model search using the specified parameters. It
#' launches the search process and monitors its progress.
#'
#' @param InterpreterPath Path to the Python interpreter executable.
#' @param Flags Flags to pass to the Python interpreter. Refer to Python
#'   documentation for details. Note that `-m` is essential (runs library module
#'   as a script and terminates option list).
#' @param DirectoryPath Optional path to the directory containing template file,
#'   tokens file and options file. If that argument is given, it overrides the
#'   paths of `TemplatePath`, `TokensPath`, `OptionsPath` with warning.
#'   Default is current working directory.
#' @param TemplatePath Path to the template file.
#' @param TokensPath Path to the tokens JSON file.
#' @param OptionsPath Path to the options JSON file.
#' @param Wait Logical. If `TRUE`, the function waits for the search process to
#'   complete and returns the results. If `FALSE`, the process is launched
#'   and exits immediately.
#'
#' @return If `Wait` is `TRUE`, a list containing the results of the search,
#'   including a data frame for all models executed, a final model and
#'   properties of the final model generated by pyDarwin when available. If
#'   nothing is available, `messages.txt` file text is given. If `Wait` is
#'   `FALSE`, `messages.txt` file location is returned where raw output of
#'   pyDarwin run is stored.
#'
#' @examples
#' \dontrun{
#' result <- run_pyDarwin(
#'   InterpreterPath = "~/darwin/venv/bin/python",
#'   DirectoryPath = "~/project_folder",
#'   TemplatePath = "template.txt",
#'   TokensPath = "tokens.json",
#'   OptionsPath = "options.json"
#' )
#' }
#'
#' @export
run_pyDarwin <- function(InterpreterPath,
                         Flags = c("-u", "-m"),
                         DirectoryPath = ".",
                         TemplatePath = "template.txt",
                         TokensPath = "tokens.json",
                         OptionsPath = "options.json",
                         Wait = TRUE) {
  if (missing(InterpreterPath)) {
    stop("Cannot start without InterpreterPath specified.")
  } else if (!file.exists(InterpreterPath)) {
    stop("InterpreterPath ", InterpreterPath, " not found.")
  }

  stopifnot(is.logical(Wait))

  if (methods::hasArg(DirectoryPath)) {
    if (!dir.exists(DirectoryPath)) {
      stop("Given directory does not exist: ", DirectoryPath)
    }

    if (TemplatePath != basename(TemplatePath)) {
      warning(
        "Since DirectoryPath argument provides a valid path,\n",
        "path to TemplatePath will be ignored"
      )
    }

    TemplatePath <-
      file.path(DirectoryPath, basename(TemplatePath))


    if (TokensPath != basename(TokensPath)) {
      warning(
        "Since DirectoryPath argument provides a valid path,\n",
        "path to TokensPath will be ignored"
      )
    }

    TokensPath <- file.path(DirectoryPath, basename(TokensPath))

    if (OptionsPath != basename(OptionsPath)) {
      warning(
        "Since DirectoryPath argument provides a valid path,\n",
        "path to OptionsPath will be ignored"
      )
    }

    OptionsPath <-
      file.path(DirectoryPath, basename(OptionsPath))
  }

  if (!file.exists(TemplatePath)) {
    stop("TemplatePath ", TemplatePath, " not found.")
  }

  if (!file.exists(TokensPath)) {
    stop("TokensPath ", TokensPath, " not found.")
  }

  if (!file.exists(OptionsPath)) {
    stop("OptionsPath ", OptionsPath, " not found.")
  }

  pyDarwinOptions <- jsonlite::fromJSON(OptionsPath)
  ProjectDirAlias <- dirname(OptionsPath)
  working_dir <-
    normalizePath(
      gsub(
        "{project_dir}",
        ProjectDirAlias,
        pyDarwinOptions$working_dir,
        fixed = TRUE
      ),
      mustWork = FALSE
    )
  MessagesFile <- file.path(working_dir, "messages.txt")
  output_dir <-
    gsub("{project_dir}",
         ProjectDirAlias,
         pyDarwinOptions$output_dir,
         fixed = TRUE)
  output_dir <-
    gsub("{working_dir}", working_dir, output_dir, fixed = TRUE)

  GlobalOptions <- NA
  if (!is.null(pyDarwinOptions$use_system_options) &&
      pyDarwinOptions$use_system_options &&
      file.exists(Sys.getenv("PYDARWIN_OPTIONS"))) {
    # could be in global options
    GlobalOptions <- jsonlite::toJSON(Sys.getenv("PYDARWIN_OPTIONS"))
    if (!is.null(GlobalOptions$engine_adapter)) {
      pyDarwinOptions$engine_adapter <- GlobalOptions$engine_adapter
    }
  } else if (is.null(pyDarwinOptions$engine_adapter)) {
    pyDarwinOptions$engine_adapter <- "nonmem"
  }

  if (.Platform$OS.type == "unix" &
      pyDarwinOptions$engine_adapter == "nlme") {
    if (!is.na(GlobalOptions) &&
        length(GlobalOptions$nlme_dir) > 0) {
      nlme_dirToUse <- GlobalOptions$nlme_dir
    } else {
      nlme_dirToUse <- pyDarwinOptions$nlme_dir
    }

    if (!is.na(GlobalOptions) &&
        length(GlobalOptions$model_run_man) > 0) {
      model_run_manToUse <- GlobalOptions$model_run_man
    } else {
      model_run_manToUse <- pyDarwinOptions$model_run_man
    }

    if (model_run_manToUse == "darwin.LocalRunManager") {
      TDL5Path <- file.path(nlme_dirToUse,
                            Sys.getenv("PML_BIN_DIR"),
                            "TDL5")
      if (file.exists(TDL5Path)) {
        # for some Linux systems the license is not working
        # for the first call
        system2(TDL5Path,
                args = "-license_check",
                stderr = FALSE,
                stdout = FALSE)
      }
    }
  }

  message(paste(
    InterpreterPath,
    paste(Flags, collapse = " "),
    "darwin.run_search",
    shQuote(TemplatePath, type = "cmd"),
    shQuote(TokensPath, type = "cmd"),
    shQuote(OptionsPath, type = "cmd")
  ))

  if (pyDarwinOptions$engine_adapter == "nonmem") {
    if (Wait) {
      ProcessValue <- system2(
        command = InterpreterPath,
        args = c(
          Flags,
          "darwin.run_search",
          shQuote(TemplatePath, type = "cmd"),
          shQuote(TokensPath, type = "cmd"),
          shQuote(OptionsPath, type = "cmd")
        ),
        wait = Wait
      )
    } else {
      # something weird happens with NONMEM when using system2
      # need to pipe files to make it work
      if (!dir.exists(working_dir)) {
        dir.create(working_dir, recursive = TRUE)
      }

      ProcessValue <- system2(
        InterpreterPath,
        args = c(
          Flags,
          "darwin.run_search",
          TemplatePath,
          TokensPath,
          OptionsPath
        ),
        stdout = file.path(working_dir, "err1.txt"),
        stderr = file.path(working_dir, "err2.txt"),
        wait = Wait
      )

    }
  } else {
    # nlme
    # cannot use sys or processx due to broken piping
    ProcessValue <- system2(
      command = InterpreterPath,
      args = c(
        Flags,
        "darwin.run_search",
        shQuote(TemplatePath, type = "cmd"),
        shQuote(TokensPath, type = "cmd"),
        shQuote(OptionsPath, type = "cmd")
      ),
      wait = Wait
    )
  }

  if (ProcessValue != 0) {
    stop(
      "Python call finished with exit code ",
      ProcessValue,
      ". Please inspect ",
      MessagesFile,
      " file for possible reasons or try to run pyDarwin directly:\n",
      paste(
        InterpreterPath,
        paste(Flags, collapse = " "),
        "darwin.run_search",
        shQuote(TemplatePath, type = "cmd"),
        shQuote(TokensPath, type = "cmd"),
        shQuote(OptionsPath, type = "cmd")
      )
    )
  }

  if (Wait) {
    ReturnedList <- list(
      results = data.frame(),
      FinalResultFile = character(),
      FinalControlFile = character()
    )

    NotFoundFiles <- c()
    resultsPath <- file.path(output_dir, "results.csv")
    if (file.exists(resultsPath)) {
      ReturnedList$results <- utils::read.csv(resultsPath)
    } else {
      NotFoundFiles <- c(NotFoundFiles, resultsPath)
    }

    if (pyDarwinOptions$engine_adapter == "nlme") {
      FinalResultFilePath <-
        file.path(output_dir, "FinalResultFile.txt")
      FinalControlFilePath <-
        file.path(output_dir, "FinalControlFile.mmdl")
    } else {
      # nonmem is default
      FinalResultFilePath <-
        file.path(output_dir, "FinalResultFile.lst")
      FinalControlFilePath <-
        file.path(output_dir, "FinalControlFile.mod")
    }

    if (file.exists(FinalResultFilePath)) {
      ReturnedList$FinalResultFile <-
        readLines(FinalResultFilePath, warn = FALSE)
    } else {
      NotFoundFiles <- c(NotFoundFiles, FinalResultFilePath)
    }

    if (file.exists(FinalControlFilePath)) {
      ReturnedList$FinalControlFile <-
        readLines(FinalControlFilePath, warn = FALSE)
    } else {
      NotFoundFiles <- c(NotFoundFiles, FinalControlFilePath)
    }

    if (length(NotFoundFiles) > 0) {
      warning(
        "The following files are not found in output directory:\n",
        paste(NotFoundFiles, collapse = ", ")
      )
    }

    if (length(ReturnedList) == 0) {
      if (!file.exists(MessagesFile)) {
        stop("Messages file ", MessagesFile, " not found.")
      }

      warning(
        "Ouput with the final results is not found in the ouput.\n",
        "Returning the messages output."
      )
      readLines(MessagesFile, warn = FALSE)
    } else {
      ReturnedList
    }
  } else {
    MessagesFile
  }
}

#' Stop pyDarwin Model Search
#'
#' This function stops a pyDarwin model search.
#'
#' @param InterpreterPath Path to the Python interpreter executable.
#' @param Flags Flags to pass to the Python interpreter. Refer to Python
#'   documentation for details. Note that `-m` is essential (runs library module
#'   as a script and terminates option list).
#' @param ForceStop Logical. If `TRUE`, `-f` flag is added to force stop search
#'   immediately.- Default is `FALSE`.
#' @param DirectoryPath the `DirectoryPath` argument of [run_pyDarwin()] or the
#'   parent folder of options file passed to that function.
#'   Default is current working directory.
#'
#' @return Returned code of [system2()] call.
#'
#' @examples
#' \dontrun{
#' stop_pyDarwin(
#'   InterpreterPath = "~/darwin/venv/bin/python",
#'   DirectoryPath = "~/project_folder")
#' }
#'
#' @export
stop_pyDarwin <- function(InterpreterPath,
                          Flags = c("-u", "-m"),
                          ForceStop = FALSE,
                          DirectoryPath = ".") {
  if (missing(InterpreterPath)) {
    stop("Cannot start without InterpreterPath specified.")
  } else if (!file.exists(InterpreterPath)) {
    stop("InterpreterPath ", InterpreterPath, " not found.")
  }

  stopifnot(is.logical(ForceStop))
  DirectoryPath <- shQuote(normalizePath(DirectoryPath), type = "cmd")

  Args <- DirectoryPath
  if (ForceStop) {
    Args <- c("-f", Args)
  }

  Args <- c(Flags, "darwin.stop_search", Args)

  system2(
    InterpreterPath,
    args = Args,
    stdout = TRUE,
    stderr = TRUE
  )
}
