% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_csr.R
\name{CSRPeaksFilter}
\alias{CSRPeaksFilter}
\title{Performs the peak selection based on complete spatial randomness test.}
\usage{
CSRPeaksFilter(
  msiData,
  method = "ClarkEvans",
  covariateImage = NULL,
  adjMethod = "bonferroni",
  returnQvalues = TRUE,
  plotCovariate = FALSE,
  cores = 1,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{msiData}{\link{msi.dataset-class} object. See \link{msiDataset}.}

\item{method}{string (default = \code{"ClarkEvans"}). CSR statistical test
applied to the peaks signal. Accepted values are:
\itemize{
   \item "ClarkEvans": performs a test based on the Clark and Evans aggregation
   R index. This test evaluates the compares of the nearest-neighbors distances
   to the case of purely random pattern.
   \item "KS": performs a test of goodness-of-fit between the signal pixels
   associated point process pattern and a spatial covariate using the
   Kolmogorov-Smirnov test. The covariate is defined by the reference image.
}}

\item{covariateImage}{\link{ms.image-class} object. An image used as covariate
(required for Kolmogorov-Smirnov test).}

\item{adjMethod}{string (default = \code{"bonferroni"}). Multiple testing correction
method. Possible values coincide with those of the \code{stats::p.adjust} function.}

\item{returnQvalues}{logical (default = \code{TRUE}). Whether the computed
q-values should be returned together with the p-values.}

\item{plotCovariate}{logical (default = \code{FALSE}). Whether the covariate image
should be visualized. Read only when \code{method = "KS"}.}

\item{cores}{integer (default = 1). Number of CPU cores. Parallel computation if
greater than 1.}

\item{verbose}{logical (default = \code{TRUE}). Additional output texts are
generated.}

\item{...}{additional parameters compatible with the \code{statspat.core} functions.
See \link[spatstat.explore]{cdf.test} for "KS" and \link[spatstat.explore]{clarkevans.test}.
for "ClarkEvans"}
}
\value{
List of the p-values and adjusted p-values for the CSR test.
}
\description{
\code{CSRPeaksFilter} returns the significance for the null hypothesis that the
spatial distribution of the peak intensities follow a random pattern. A
significant p-value (q-values can be returned after applying multiple testing
correction) allows to reject the hypothesis that the spatial distribution of
a peak signal is random. The tests are performed using the functions available
in the \code{statspat} R package.
}
\examples{
## Load package
library("SPUTNIK")

## Mass spectrometry intensity matrix
X <- matrix(rnorm(16000), 400, 40)
X[X < 0] <- 0

## Print original dimensions
print(dim(X))

## m/z vector
mzVector <- seq(600, 900, by = (900 - 600) / 39)

## Read the image size
imSize <- c(20, 20)

## Construct the ms.dataset object
msiX <- msiDataset(X, mzVector, imSize[1], imSize[2])

## Calculate the p-values using the Clark Evans test, then apply Benjamini-
## Hochberg correction.
csr <- CSRPeaksFilter(
  msiData = msiX, method = "ClarkEvans",
  calculateCovariate = FALSE, adjMethod = "BH"
)

## Print selected peaks
print(csr$q.value)

## Create a new filter selecting corrected p-values < 0.001
selIdx <- which(csr$q.value < 0.001)
csrFilter <- createPeaksFilter(selIdx)
}
\references{
Baddeley, A., & Turner, R. (2005). Spatstat: an R package for
analyzing spatial point patterns. Journal of statistical software, 12(6), 1-42.

Clark, P.J. and Evans, F.C. (1954) Distance to nearest neighbour
as a measure of spatial relationships in populations. Ecology 35, 445–453.

Berman, M. (1986) Testing for spatial association between a point
process and another stochastic process. Applied Statistics 35, 54–62.
}
\author{
Paolo Inglese \email{p.inglese14@imperial.ac.uk}
}
