% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lik_profile.R
\name{lik_profile}
\alias{lik_profile}
\title{Likelihood profiling}
\usage{
lik_profile(
  x,
  par,
  output,
  data = NULL,
  bounds = NULL,
  refit = TRUE,
  type = c("coarse", "fine"),
  break_prof = FALSE,
  ...
)
}
\arguments{
\item{x}{either a single \link{scenario} or a list of \link{caliset} objects}

\item{par}{named vector - parameters (names and values) to be profiled}

\item{output}{character vector, name of output column of \code{\link[=simulate]{simulate()}} that
is used in calibration}

\item{data}{only needed if \code{x} is a \link{scenario}}

\item{bounds}{optional list of lists (including lower and upper bound): uses defaults in \code{x} object, but
can be overwritten here (e.g. bounds <- list(k_resp = list(0,10), k_phot_max = list(0,30)) )}

\item{refit}{if 'TRUE' (default), refit if a better minimum is found}

\item{type}{"fine" or "coarse" (default) likelihood profiling}

\item{break_prof}{if 'TRUE' (default), stop the profiling if a better optimum is located}

\item{...}{additional parameters passed on to \code{\link[stats:optim]{stats::optim()}} and \code{\link[=calibrate]{calibrate()}}}
}
\value{
A list containing, for each parameter profiled, the likelihood
profiling results as a dataframe;
the 95\% confidence interval; the original parameter value; the likelihood plot object; and
the recalibrated parameter values (in case a lower optimum was found)
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

The aim of the function is two-fold: 1) estimate a 95\% confidence
around each parameter of a calibrated model, and 2) see if perhaps a local minimum was found rather
than a global minimum. To achieve this, the likelihood profiling goes through
every parameter one by one. For each parameter,
the model is sequentially refit with the parameter value set to
increasingly lower and higher values, and the likelihood of the model given the
data calculated (using \code{\link[=log_lik]{log_lik()}}). The likelihood is then compared
to the likelihood of the original model (using a likelihood ratio). This leads
to the development of a likelihood profile, from which a plot a 95\%
confidence interval for the parameter is derived.

The idea of the function is a variable stepwise algorithm:
When the likelihood ratio changes very little (less than \code{l_crit_min}), the stepsize is
increased (up to a maximum, specified by \code{f_step_max}). When the lik.
ratio changes too much (more than \code{l_crit_max}), the algorithm tries again
with a smaller stepsize (also bound to a minimum: \code{f_step_min}). Note that
the stepsize is used as a fraction of the parameter value that is tried.
To prevent very small stepsizes when the value goes towards zero (as can
be the case for effect thresholds), an absolute minimum
stepsize (\code{f_step_abs}), which is specified as a fraction of the best
parameter value (\code{Xhat}) (unless it is zero, then algoritm takes
something small).

The function was inspired by a MatLab BYOM v.6.8 procedure, created by
Tjalling Jager. For details, please refer to BYOM (http://debtox.info/byom.html)
as well as Jager (2021).
}
\examples{
# Example with Lemna model - physiological params
library(dplyr)

# exposure - control run
exp <- Schmitt2013 \%>\%
  filter(ID == "T0") \%>\%
  select(time=t, conc)

# observations - control run
obs <- Schmitt2013 \%>\%
  filter(ID == "T0") \%>\%
   select(t, BM=obs)

# update metsulfuron
myscenario <- metsulfuron \%>\%
  set_param(c(k_phot_fix = TRUE,Emax = 1)) \%>\%
  set_init(c(BM = 12)) \%>\%
  set_exposure(exp)

fit <- calibrate(
  x = myscenario,
  par = c(k_phot_max = 1),
  data = obs,
  output = "BM",
  lower=0,
  upper=1,
  method="Brent"
)

# Likelihood profiling
\donttest{
res <- lik_profile(
  x = myscenario,
  data = obs,
  output = "BM",
  par = fit$par,
  bounds = list(
    k_phot_max = list(0, 30)
  ),
  refit = FALSE,
  type = "fine",
  method = "Brent"
)
# plot
plot_lik_profile(res)
}

}
\references{
Jager T, 2021. Robust Likelihood-Based Optimization and Uncertainty Analysis
of Toxicokinetic-Toxicodynamic Models. Integrated Environmental Assessment and
Management 17:388-397. \doi{10.1002/ieam.4333}
}
