% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lgp.R
\name{lgp}
\alias{lgp}
\title{Linked (D)GP emulator construction}
\usage{
lgp(struc, emulators = NULL, B = 10, activate = TRUE, verb = TRUE, id = NULL)
}
\arguments{
\item{struc}{the structure of the linked emulator, which can take one of two forms:
\itemize{
\item \ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} a list contains \emph{L} (the number of layers in a systems of computer models) sub-lists,
each of which represents a layer and contains (D)GP emulators (represented by
instances of S3 class \code{gp} or \code{dgp}) of computer models. The sub-lists are placed in the list
in the same order of the specified computer model system's hierarchy. \strong{This option is deprecated and will be removed in the next release.}
\item \ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} a data frame that defines the connection structure between emulators in the linked system, with the following columns:
\itemize{
\item \code{From_Emulator}: the ID of the emulator providing the output. This ID must match the \code{id} slot
in the corresponding emulator object (produced by \code{\link[=gp]{gp()}} or \code{\link[=dgp]{dgp()}}) within \code{emulators} argument of \code{\link[=lgp]{lgp()}}, or it should
be special value \code{"Global"}, indicating the global inputs to the model chain or network. The \code{id} slot
is either automatically generated by \code{\link[=gp]{gp()}} or \code{\link[=dgp]{dgp()}}, or can be manually specified via the \code{id} argument in these functions or set with the
\code{\link[=set_id]{set_id()}} function.
\item \code{To_Emulator}: the ID of the emulator receiving the input, also matching the \code{id} slot in the
corresponding emulator object.
\item \code{From_Output}: a single integer specifying the output dimension of the \code{From_Emulator} that is being connected to the
input dimension of the \code{To_Emulator} specified by \code{To_Input}. If \code{From_Emulator} is \code{"Global"}, then \code{From_Output}
indicates the dimension of the global input passed to the \code{To_Emulator}.
\item \code{To_Input}: a single integer specifying the input dimension of the \code{To_Emulator} that is receiving the \code{From_Output}  dimension
from the \code{From_Emulator}.
}

Each row represents a single one-to-one connection between a specified output dimension of \code{From_Emulator}
and a corresponding input dimension of \code{To_Emulator}. If multiple connections are required between
two emulators, each connection should be specified in a separate row.

\strong{Note:} When using the data frame option for \code{struc}, the \code{emulators} argument must be provided.
}}

\item{emulators}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} a list of emulator objects, each containing an \code{id} slot that uniquely identifies it within the
linked system. The \code{id} slot in each emulator object must match the \code{From_Emulator}/\code{To_Emulator} columns in \code{struc}.

If the same emulator is used multiple times within the linked system, the list must contain distinct copies
of that emulator, each with a unique ID stored in their \code{id} slot. Use the \code{\link[=set_id]{set_id()}} function to produce copies with different IDs
to ensure each instance can be uniquely referenced.}

\item{B}{the number of imputations used for prediction. Increase the value to refine representation of
imputation uncertainty. If the system consists of only GP emulators, \code{B} is set to \code{1} automatically. Defaults to \code{10}.}

\item{activate}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} a bool indicating whether the initialized linked emulator should be activated:
\itemize{
\item If \code{activate = FALSE}, \code{\link[=lgp]{lgp()}} returns an inactive linked emulator, allowing inspection of its structure using \code{\link[=summary]{summary()}}.
\item If \code{activate = TRUE}, \code{\link[=lgp]{lgp()}} returns an active linked emulator, ready for prediction and validation using \code{\link[=predict]{predict()}} and \code{\link[=validate]{validate()}}, respectively.
}

Defaults to \code{TRUE}. This argument is only applicable when \code{struc} is specified as a data frame.}

\item{verb}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} a bool indicating if the trace information on linked (D)GP emulator construction should be printed during the function call.
Defaults to \code{TRUE}. This argument is only applicable when \code{struc} is specified as a data frame.}

\item{id}{an ID to be assigned to the linked (D)GP emulator. If an ID is not provided (i.e., \code{id = NULL}), a UUID
(Universally Unique Identifier) will be automatically generated and assigned to the emulator. Defaults to \code{NULL}.}
}
\value{
An S3 class named \code{lgp} that contains three slots:
\itemize{
\item \code{id}: A number or character string assigned through the \code{id} argument.
\item \code{constructor_obj}: a list of 'python' objects that stores the information of the constructed linked emulator.
\item \code{emulator_obj}, a 'python' object that stores the information for predictions from the linked emulator.
\item \code{specs}: a list that contains
\enumerate{
\item \code{seed}: the random seed generated to produce the imputations. This information is stored for reproducibility
when the linked (D)GP emulator (that was saved by \code{\link[=write]{write()}} with the light option \code{light = TRUE}) is loaded back
to R by \code{\link[=read]{read()}}.
\item \code{B}: the number of imputations used to generate the linked (D)GP emulator.
}

\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#new}{\figure{lifecycle-new.svg}{options: alt='[New]'}}}{\strong{[New]}} If \code{struc} is a data frame, \code{specs} also includes:
\enumerate{
\item \code{metadata}: a data frame providing configuration details for each emulator in the linked system, with following columns:
\itemize{
\item \code{Emulator}: the ID of an emulator.
\item \code{Layer}: the layer in the linked system where the emulator is positioned. A lower \code{Layer} number indicates
a position closer to the input, with layer numbering increasing as you move away from the input.
\item \code{Pos_in_Layer}: the position of the emulator within its layer. A lower \code{Pos_in_Layer} number
indicates a position higher up in that layer.
\item \code{Total_Input_Dims}: the total number of input dimensions of the emulator.
\item \code{Total_Output_Dims}: the total number of output dimensions of the emulator.
}
\item \code{struc}: The linked system structure, as supplied by \code{struc}.
}
}

The returned \code{lgp} object can be used by
\itemize{
\item \code{\link[=predict]{predict()}} for linked (D)GP predictions.
\item \code{\link[=validate]{validate()}} for OOS validation.
\item \code{\link[=plot]{plot()}} for validation plots.
\item \code{\link[=summary]{summary()}} to summarize the constructed linked (D)GP emulator.
\item \code{\link[=write]{write()}} to save the linked (D)GP emulator to a \code{.pkl} file.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#updated}{\figure{lifecycle-updated.svg}{options: alt='[Updated]'}}}{\strong{[Updated]}}

This function constructs a linked (D)GP emulator for a model chain or network.
}
\details{
See further examples and tutorials at \url{https://mingdeyu.github.io/dgpsi-R/}.
}
\examples{
\dontrun{

# load the package and the Python env
library(dgpsi)

# model 1
f1 <- function(x) {
  (sin(7.5*x)+1)/2
}
# model 2
f2 <- function(x) {
  2/3*sin(2*(2*x - 1))+4/3*exp(-30*(2*(2*x-1))^2)-1/3
}
# linked model
f12 <- function(x) {
  f2(f1(x))
}

# training data for Model 1
X1 <- seq(0, 1, length = 9)
Y1 <- sapply(X1, f1)
# training data for Model 2
X2 <- seq(0, 1, length = 13)
Y2 <- sapply(X2, f2)

# emulation of model 1
m1 <- gp(X1, Y1, name = "matern2.5", id = "emulator1")
# emulation of model 2
m2 <- dgp(X2, Y2, depth = 2, name = "matern2.5", id = "emulator2")

struc <- data.frame(From_Emulator = c("Global", "emulator1"),
                    To_Emulator = c("emulator1", "emulator2"),
                    From_Output = c(1, 1),
                    To_Input = c(1, 1))
emulators <- list(m1, m2)

# construct the linked emulator for visual inspection
m_link <- lgp(struc, emulators, activate = FALSE)

# visual inspection
summary(m_link)

# build the linked emulator for prediction
m_link <- lgp(struc, emulators, activate = TRUE)
test_x <- seq(0, 1, length = 300)
m_link <- predict(m_link, x = test_x)

# OOS validation
validate_x <- sample(test_x, 20)
validate_y <- sapply(validate_x, f12)
plot(m_link, validate_x, validate_y, style = 2)

# write and read the constructed linked emulator
write(m_link, 'linked_emulator')
m_link <- read('linked_emulator')
}
}
