% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_fairness_metrics.R
\name{get_fairness_metrics}
\alias{get_fairness_metrics}
\title{Compute Fairness Metrics for Binary Classification}
\usage{
get_fairness_metrics(
  data,
  outcome,
  group,
  group2 = NULL,
  condition = NULL,
  probs,
  confint = TRUE,
  cutoff = 0.5,
  bootstraps = 2500,
  alpha = 0.05,
  digits = 2
)
}
\arguments{
\item{data}{A data frame containing the outcome, group, and predicted probabilities.}

\item{outcome}{The name of the column containing the true binary outcome.}

\item{group}{The name of the column representing the sensitive attribute (e.g., race, gender).}

\item{group2}{Define if conditional statistical parity is desired. Name of a secondary group variable used for conditional fairness analysis.}

\item{condition}{Define if conditional statistical parity is desired. If the conditional group is categorical, the condition
supplied must be a character of the levels to condition on. If the conditional
group is continuous, the conditions supplied must be a character containing
the sign of the condition and the value to threshold the continuous variable
(e.g. "<50", ">50", "<=50", ">=50").}

\item{probs}{The name of the column with predicted probabilities.}

\item{confint}{Logical indicating whether to calculate confidence intervals.}

\item{cutoff}{Numeric threshold for classification. Default is 0.5.}

\item{bootstraps}{Number of bootstrap samples. Default is 2500.}

\item{alpha}{Significance level for confidence intervals. Default is 0.05.}

\item{digits}{Number of digits to round the metrics to. Default is 2.}
}
\value{
A list containing:
\describe{
\item{performance}{Data frame with performance metrics by group.}
\item{fairness}{Data frame with computed fairness metrics and optional confidence intervals.}
}
}
\description{
Computes a comprehensive set of fairness metrics for binary classification models, disaggregated by a sensitive attribute (e.g., race, gender). Optionally, conditional fairness can be evaluated using a second attribute and a specified condition. The function also computes corresponding performance metrics used in the fairness calculations.
}
\details{
The results are returned as a list of two data frames:
\itemize{
\item \code{performance}: Contains performance metrics (e.g., TPR, FPR, PPV) by group.
\item \code{fairness}: Contains group-level fairness metrics (e.g., disparities or ratios), confidence intervals (if specified).
}
\subsection{Fairness Metrics Included:}{
\itemize{
\item \strong{Statistical Parity}: Difference in positive prediction rates across groups.
\item \strong{Conditional Statistical Parity} \emph{(if group2 and condition are specified)}:
Parity conditioned on a second group and value.
\item \strong{Equal Opportunity}: Difference in true positive rates (TPR) across groups.
\item \strong{Predictive Equality}: Difference in false positive rates (FPR) across groups.
\item \strong{Balance for Positive Class}: Checks whether the predicted probability distributions for
positive outcomes are similar across groups.
\item \strong{Balance for Negative Class}: Same as above, but for negative outcomes.
\item \strong{Positive Predictive Parity}: Difference in positive predictive values (precision) across groups.
\item \strong{Negative Predictive Parity}: Difference in negative predictive values across groups.
\item \strong{Brier Score Parity}: Difference in Brier scores across groups.
\item \strong{Overall Accuracy Parity}: Difference in overall accuracy across groups.
\item \strong{Treatment Equality}: Ratio of false negatives to false positives across groups.
}
}
}
\examples{
\donttest{
library(fairmetrics)
library(dplyr)
library(randomForest)
library(magrittr)
data("mimic_preprocessed")
set.seed(123)
train_data <- mimic_preprocessed \%>\%
  dplyr::filter(dplyr::row_number() <= 700)
# Fit a random forest model
rf_model <- randomForest::randomForest(factor(day_28_flg) ~ ., data = train_data, ntree = 1000)
# Test the model on the remaining data
test_data <- mimic_preprocessed \%>\%
  dplyr::mutate(gender = ifelse(gender_num == 1, "Male", "Female"))\%>\%
  dplyr::filter(dplyr::row_number() > 700)

test_data$pred <- predict(rf_model, newdata = test_data, type = "prob")[, 2]

# Fairness evaluation
# We will use sex as the sensitive attribute and day_28_flg as the outcome.
# We choose threshold = 0.41 so that the overall FPR is around 5\%.

# Get Fairness Metrics
get_fairness_metrics(
 data = test_data,
 outcome = "day_28_flg",
 group = "gender",
 group2 = "age",
 condition = ">=60",
 probs = "pred",
 confint = TRUE,
 cutoff = 0.41,
 alpha = 0.05
)
}


}
