#' Plot Spearman’s Rho Autocorrelation
#'
#' Visualizes Spearman’s rho serial correlation coefficients with shaded points 
#' indicating statistical significance.
#'
#' @param results A list generated by [eda_spearman_test()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot showing:
#' - Vertical segments from \eqn{y=0} up to each \eqn{\rho} value at its lag.
#' - Filled circles at each lag, filled black if serial correlation is detected.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' results <- eda_spearman_test(data)
#' plot_spearman_test(results)
#'
#' @import ggplot2
#' @export

plot_spearman_test <- function(results, ...) {

	# Create dataframe for generating the results
	rho_df <- data.frame(
		lag = 1:length(results$rho),
		rho = results$rho,
		significant = results$significant
	)

	# Capture optional arguments
	args <- list(...)

    # Set default values
    title <- args$title %||% "Spearman's Rho Autocorrelation"
    xlabel <- args$xlabel %||% "Lag"
    ylabel <- args$ylabel %||% "Spearman's Rho"

	# First subplot: Spearman's Rho Autocorrelation
	p1 <- ggplot(rho_df, aes(x = .data$lag, y = .data$rho)) +
		geom_segment(
			aes(x = .data$lag, xend = .data$lag, y = 0, yend = .data$rho)
		) +
		geom_point(
			aes(fill = .data$significant),
			shape = 21,
			size = 3,
			stroke = 1.2
		) +
		labs(title = title, x = xlabel, y = ylabel, fill = "Legend") + 
		scale_fill_manual(
			values = c(`TRUE` = "black", `FALSE` = "white"),
			labels = c("No Serial Correlation", "Serial Correlation"),
		)

	# Return the plot with added theme
	add_theme(add_scales(p1))

}

 
