% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04-crop-analysis.R
\name{analyze_cdl_crops_dynamic}
\alias{analyze_cdl_crops_dynamic}
\title{Analyze CDL crops dynamically}
\usage{
analyze_cdl_crops_dynamic(
  cdl_data,
  crop_selection,
  region_boundary = NULL,
  analysis_type = "classify",
  output_folder = NULL,
  year = NULL
)
}
\arguments{
\item{cdl_data}{CDL raster data (file path, directory, or SpatRaster object)}

\item{crop_selection}{Crop selection (names, codes, or categories)}

\item{region_boundary}{Region boundary}

\item{analysis_type}{Type of analysis: "mask", "classify", "area"}

\item{output_folder}{Output directory}

\item{year}{Year for analysis (optional)}
}
\value{
Analysis results depend on analysis_type:
\itemize{
\item \strong{"mask"}: SpatRaster with binary mask (1 = crop, 0 = other)
\item \strong{"classify"}: SpatRaster with binary classification
\item \strong{"area"}: List with detailed area analysis:
\itemize{
\item \code{crop_areas_ha}: Named list of areas by crop code (hectares)
\item \code{total_area_ha}: Total crop area (hectares)
\item \code{pixel_size_ha}: Individual pixel size (hectares)
\item \code{crop_codes}: CDL codes analyzed
\item \code{crop_name}: Crop name(s)
\item \code{region}: Region analyzed
\item \code{raster_info}: Technical raster information
\item \code{total_pixels}: Total number of pixels
\item \code{valid_pixels}: Number of valid (non-NA) pixels
}
}
}
\description{
Perform dynamic analysis of CDL crop data including area calculations,
classification, and masking for any crop combination. Now accepts file paths directly.
}
\details{
\subsection{Usage Tips:}{
\subsection{Area Analysis Results:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Access specific results
result$total_area_ha          # Total area in hectares
result$crop_areas_ha          # Area by crop code
result$crop_areas_ha[["5"]]   # Soybean area (code 5)
}\if{html}{\out{</div>}}
}

\subsection{Visualization:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# For mask/classify results (SpatRaster):
terra::plot(result)           # Plot the raster

# For area results (list):
print(result$total_area_ha)   # Print total area
barplot(unlist(result$crop_areas_ha))  # Simple bar plot
}\if{html}{\out{</div>}}
}

}
}
\examples{
\dontrun{
# These examples require actual CDL data files
# Analyze soybean area in Iowa - accepts file path directly
soybean_area <- analyze_cdl_crops_dynamic(
  "/path/to/cdl_2023.tif", "soybeans", "Iowa", "area"
)

# Access area results
soybean_area$total_area_ha                    # Total hectares
soybean_area$crop_areas_ha[["5"]]            # Soybean area (code 5)
soybean_area$total_area_ha * 2.47105         # Convert to acres

# Create grain classification
grain_classes <- analyze_cdl_crops_dynamic(
  cdl_data, "grains", "CONUS", "classify"
)
terra::plot(grain_classes)  # Plot the classification

# Works with directories too
results <- analyze_cdl_crops_dynamic(
  "/path/to/cdl/files/", "corn", "Ohio", "area"
)
}

\donttest{
# Example with mock CDL data (this can run)
# Create sample CDL raster
mock_cdl <- terra::rast(nrows = 10, ncols = 10, xmin = 0, xmax = 10,
                       ymin = 0, ymax = 10, crs = "EPSG:4326")
terra::values(mock_cdl) <- sample(c(1, 5, 24), 100, replace = TRUE)  # corn, soy, wheat

# Analyze mock data
result <- analyze_cdl_crops_dynamic(mock_cdl, "corn", analysis_type = "mask")
print(class(result))  # Should be SpatRaster
}

}
