#' Visual Parameters for ggparallel Plots
#'
#' Configures aesthetic and layout settings for plots generated by `ggparallel`.
#'
#' @param show_legend Display the legend on the plot (flag).
#' @param show_legend_titles Display titles for legends (flag).
#' @param legend_position Position of the legend ("right", "left", "bottom", "top").
#' @param legend_title_position Position of the legend title ("top", "bottom", "left", "right").
#' @param legend_nrow Number of rows in the legend (number).
#' @param legend_ncol Number of columns in the legend. If set, `legend_nrow` should be `NULL` (number).
#' @param legend_key_size Size of the legend key symbols. (number).
#' @param beautify_text Beautify y-axis text and legend titles by capitalizing words and adding spaces (flag).
#' @param fontsize_x_axis_text fontsize of the x-axis text describing column names (number)
#' @param max_digits_bounds Number of digits to round the axis bounds label text to (number)
#' @param line_alpha Alpha of line geom (number)
#' @param line_width Width of the line geom (number)
#' @param line_type Type of line geom (number or string. see [ggplot2::aes_linetype_size_shape()] for valid options)
#' @param x_axis_gridlines Customise look of x axis gridlines. Must be either a call to [ggplot2::element_line()] or [ggplot2::element_blank()].
#' @param x_axis_text_angle Angle of the x axis text describing column names (number)
#' @param x_axis_text_hjust Horizontal Justification of the x axis text describing column names (number)
#' @param x_axis_text_vjust Vertical Justification of the x axis text describing column names (number)
#' @param show_column_names Show column names as x axis text (flag)
#' @param show_points Show points (flag)
#' @param show_bounds_labels Show bounds (min and max value) of each feature with labels above / below the axes (flag)
#' @param show_bounds_rect Show bounds (min and max value) of each feature with a rectangular graphic (flag)
#' @param interactive_svg_width,interactive_svg_height Width and height of the interactive graphic region (in inches). Only used when `interactive = TRUE`.
#' @return A list of visualization parameters for `ggparallel`.
#' @export
#' @inherit ggparallel examples
ggparallel_options <- function(

  # Legend
  show_legend = TRUE,
  show_legend_titles = FALSE,
  legend_position = c( "bottom", "right", "left", "top"),
  legend_title_position = c("left", "top", "bottom",  "right"),
  legend_nrow = NULL,
  legend_ncol = NULL,
  legend_key_size = 1,

  # Text
  beautify_text = TRUE,
  max_digits_bounds = 1,
  x_axis_text_angle = 90,
  x_axis_text_hjust = 0,
  x_axis_text_vjust = 0.5,
  fontsize_x_axis_text = 12,

  # Show / hide text or labels
  show_column_names = TRUE,
  show_points = FALSE,
  show_bounds_labels = FALSE,
  show_bounds_rect = FALSE,

  # Line
  line_alpha = 0.5,
  line_width = NULL,
  line_type = 1,

  # Gridlines
  x_axis_gridlines = ggplot2::element_line(colour = "black"),

  # Interactivity
  interactive_svg_width = NULL,
  interactive_svg_height = NULL
) {

  # Legend-related validations
  assertions::assert_flag(show_legend)
  assertions::assert_flag(show_legend_titles)
  assertions::assert_number(legend_key_size)
  if (!is.null(legend_nrow)) assertions::assert_number(legend_nrow)
  if (!is.null(legend_ncol)) assertions::assert_number(legend_ncol)
  if (!is.null(legend_ncol) & !is.null(legend_nrow)) {
    cli::cli_warn(c("!" = "Both {.arg legend_ncol} and {.arg legend_nrow} were supplied. {.arg legend_nrow} will be ignored. Explicitly set one of these arguments to NULL to avoid this warning message."))
    legend_nrow <- NULL
  }

  # Text-related validation
  assertions::assert_flag(beautify_text)
  assertions::assert_number(max_digits_bounds)
  assertions::assert_number(x_axis_text_angle)
  assertions::assert_number(x_axis_text_hjust)
  assertions::assert_number(x_axis_text_vjust)
  assertions::assert_number(fontsize_x_axis_text)

  # Show/Hide Related
  assertions::assert_flag(show_column_names)
  assertions::assert_flag(show_points)
  assertions::assert_flag(show_bounds_labels)
  assertions::assert_flag(show_bounds_rect)

  # Geometric Properties
  assertions::assert_number(line_alpha)
  assertions::assert_no_missing(line_type)
  assertions::assert_non_null(line_type)
  if(is.numeric(line_type)) assertions::assert_one_of(line_type, 1:6)
  if(is.character(line_type)) assertions::assert_one_of(line_type, c("blank", "solid", "dashed", "dotted", "dotdash", "longdash", "twodash"))
  if(!is.null(line_width)) assertions::assert_number(line_width)

  # Element Theming
  assertions::assert_class(x_axis_gridlines, "element")

  # Interactive plot dimensions validations
  if (!is.null(interactive_svg_width)) {
    assertions::assert_number(interactive_svg_width)
  }
  if (!is.null(interactive_svg_height)) {
    assertions::assert_number(interactive_svg_height)
  }

  # Match argument values
  legend_position <- rlang::arg_match(legend_position)
  legend_title_position <- rlang::arg_match(legend_title_position)

  # Create options list
  opts <- list(
    show_legend = show_legend,
    show_legend_titles = show_legend_titles,
    legend_position = legend_position,
    legend_title_position = legend_title_position,
    legend_nrow = legend_nrow,
    legend_ncol = legend_ncol,
    legend_key_size = legend_key_size,
    beautify_text = beautify_text,
    max_digits_bounds = max_digits_bounds,
    x_axis_text_angle = x_axis_text_angle,
    x_axis_text_hjust = x_axis_text_hjust,
    x_axis_text_vjust = x_axis_text_vjust,
    show_column_names = show_column_names,
    show_points = show_points,
    show_bounds_labels = show_bounds_labels,
    show_bounds_rect = show_bounds_rect,
    line_alpha = line_alpha,
    line_width = line_width,
    line_type = line_type,
    x_axis_gridlines = x_axis_gridlines,
    fontsize_x_axis_text = fontsize_x_axis_text,
    interactive_svg_width = interactive_svg_width,
    interactive_svg_height = interactive_svg_height
  )

  # Assign class for validation
  class(opts) <- "ggparallel_options"

  return(opts)
}
