% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weightedPivotCoord.R
\name{weightedPivotCoord}
\alias{weightedPivotCoord}
\title{Weighted pivot coordinates}
\usage{
weightedPivotCoord(
  x,
  pivotvar = 1,
  option = "var",
  method = "classical",
  pow = 1,
  yvar = NULL
)
}
\arguments{
\item{x}{object of class `data.frame` or `matrix`; positive values only}

\item{pivotvar}{pivotal variable; if any other number than 1, the data are resorted in that sense that 
pivotvar is shifted to the first part}

\item{option}{option for the choice of weights. If `option = "var"` (default), weights are based on variation matrix elements: 
`(1/t_{1j})^pow`, if `option = "cor"`, weights are based on correlations between variable specified in yvar and logratios and its distribution:
`|integral_{0}^{r_{j}} f(x) dx|`, `f(x)...` Kernel density estimator for `s_{j}; s_{j}=0 if |r_{j}|<cut` otherwise `s_{j}=r_{j}`,
`cut = min(#r_{j}=>0/#r_{j}, #r_{j}<0/#r_{j}`, with Gaussian Kernel function and bandwidth `h=0.05`.}

\item{method}{method for estimation of variation/correlation,
if `option = "classical"` (default), classical estimation is applied,
if `option = "robust"`, robust estimation is applied;}

\item{pow}{if `option = "var"`, power `pow` is applied on unnormalized weights; default is 1;}

\item{yvar}{if `option = "cor"`, weights are based on correlation between logratios and variable specified in `yvar`;}
}
\value{
\item{WPC}{weighted pivot coordinates (matrix with n rows and (D-1) columns)}
\item{w}{logcontrasts (matrix with D rows and (D-1) columns)}
}
\description{
Weighted pivot coordinates as a special case of isometric logratio coordinates.
}
\details{
Weighted pivot coordinates map D-part compositional data from the simplex into a (D-1)-dimensional real space isometrically.
 The relevant relative information about one of parts is contained in the first coordinate.
 Unlike in the (ordinary) pivot coordinates, the pairwise logratios aggregated into the first coordinate are weighted according to their relevance for the purpose of the analysis.
}
\examples{
###################
data(phd)
x <- phd[, 7:ncol(phd)]
x[x == 0] <- 0.1 # better: impute with one 
                 # of the zero imputation methods 
                 # from robCompositions

# first variable as pivotal, weights based on variation matrix
wpc_var <- weightedPivotCoord(x)
coordinates <- wpc_var$WPC
logcontrasts <- wpc_var$w

# third variable as pivotal, weights based on variation matrix, 
# robust estimation of variance, effect of weighting enhanced
wpc_var <- weightedPivotCoord(x, pivotvar = 3, method = "robust", pow = 2)
coordinates = wpc_var$WPC
logcontrasts = wpc_var$w

# first variable as pivotal, weights based on correlation between pairwise logratios and y
wpc_cor <- weightedPivotCoord(x, option = "cor", yvar = phd$female)
coordinates <- wpc_cor$WPC
logcontrasts <- wpc_cor$w

# fifth variable as pivotal, weights based on correlation between pairwise logratios 
# and y, robust estimation of correlation
wpc_cor <- weightedPivotCoord(x, pivotvar = 5, option = "cor", method = "robust", yvar = phd$female)
coordinates <- wpc_cor$WPC
logcontrasts <- wpc_cor$w

}
\references{
Hron K, Filzmoser P, de Caritat P, Fiserova E, Gardlo A (2017) 
Weighted 'pivot coordinates for compositional data and their application to geochemical mapping.
Mathematical Geosciences 49(6):797-814.
 
Stefelova N, Palarea-Albaladejo J, and Hron K (2021)
Weighted pivot coordinates for PLS-based marker discovery in high-throughput compositional data.
Statistical Analysis and Data Mining: The ASA Data Science Journal 14(4):315-330.
}
\seealso{
\code{\link{pivotCoord}}
}
\author{
Nikola Stefelova
}
\keyword{multivariate}
