% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elfunctions.R
\name{EL1}
\alias{EL1}
\title{Self-concordant multi-variate empirical likelihood with counts}
\source{
This original code was written for \insertCite{owen2013self}{smoothemplik}
and [published online](https://artowen.su.domains/empirical/) by Art B. Owen
(March 2015, February 2017). The present version was rewritten in \code{Rcpp} and
slightly reworked to contain fewer inner functions and loops.
}
\usage{
EL1(
  z,
  mu = NULL,
  ct = NULL,
  shift = NULL,
  lambda.init = NULL,
  renormalise = FALSE,
  return.weights = FALSE,
  lower = NULL,
  upper = NULL,
  order = NA,
  weight.tolerance = NULL,
  deriv = FALSE,
  thresh = 1e-30,
  itermax = 100L,
  verbose = FALSE,
  alpha = 0.3,
  beta = 0.8,
  backeps = 0,
  gradtol = 1e-12,
  steptol = 1e-12,
  ftol = 1e-14,
  stallmax = 5
)
}
\arguments{
\item{z}{A numeric vector or a matrix with one data vector per column.}

\item{mu}{Hypothesised mean, default \code{(0 ... 0)} in \eqn{R^{\mathrm{ncol}(z)}}{R^ncol(z)}.}

\item{ct}{Numeric count variable with non-negative values that indicates the multiplicity of observations.}

\item{shift}{The value to add in the denominator (useful in case there are extra
Lagrange multipliers): \eqn{1 + \lambda'Z + shift}{1 + lambda'Z + shift}.}

\item{lambda.init}{Starting lambda, default \code{(0 ... 0)}. Improves speed and accuracy
in sequential problems if supplied from the previous iteration.}

\item{renormalise}{If \code{FALSE}, then uses the total sum of counts as the number of observations,
like in vanilla empirical likelihood, due to formula (2.9) in \insertCite{owen2001empirical}{smoothemplik},
otherwise re-normalises the counts to 1 according to \insertCite{cosma2019inference}{smoothemplik}
(p. 170, the topmost formula).}

\item{return.weights}{Logical: if \code{TRUE}, returns the empirical probabilities.
Default is memory-saving (\code{FALSE}).}

\item{lower}{Lower cut-off for [logTaylor()], default \code{1/NROW(z)}.}

\item{upper}{Upper cut-off for [logTaylor()], default \code{Inf}.}

\item{order}{Positive even integer such that the Taylor approximation of this order to
\eqn{\log x}{log(x)} is self-concordant; usually 4 or 2. Passed to [logTaylor()].}

\item{weight.tolerance}{Weight tolerance for counts to improve numerical stability
(defaults to \code{sqrt(.Machine$double.eps)} times the maximum weight).}

\item{deriv}{Logical: if \code{TRUE}, computes and returns the first two directional derivatives
of log-ELR w.r.t. \code{mu} in the direction of the hypothesised value.}

\item{thresh}{Target tolerance on the squared Newton decrement: loop stops when \code{decr^2 <= thresh}.
(If \code{verbose} is \code{TRUE}, decrement itself is printed.)}

\item{itermax}{Maximum number of outer iterations of the damped Newton method (seems ample).}

\item{verbose}{Logical: print output diagnostics?}

\item{alpha}{Backtracking line search Armijo parameter: acceptance of a decrease in function value
by \eqn{\alpha f}{ALPHA*f} of the prediction based on the linear extrapolation. Smaller makes acceptance easier.}

\item{beta}{Backtracking step shrinkage factor in \code{[0, 1]}. 0.1 corresponds to a very crude search,
0.8 corresponds to a less crude search.}

\item{backeps}{Backtrack threshold, a small slack added to Armijo RHS: the search can miss by this much.
Accept if \eqn{f(x+tp) \le f(x)+\alpha t g'p + \mathrm{backeps}}{f(x+tp) <= f(x) + alpha\*t\*g'p + backeps}.
Consider setting it to \code{1e-10} if backtracking seems to be failing due to round-off.}

\item{gradtol}{Gradient tolerance: stop if \code{||g|| <= gradtol}.}

\item{steptol}{Step tolerance: stop if the relative size is tiny: \code{||x2-x1||/max(1, ||x2||) < ftol}.}

\item{ftol}{Function change tolerance: stop if the relative function-value change is less than \code{ftol}.}

\item{stallmax}{Stop if both \code{rel_step <= steptol} and \code{rel_f <= ftol} hold for this many consecutive iterations.}
}
\value{
A list with the following values:
\describe{
    \item{logelr}{Log of empirical likelihood ratio (equal to 0 if the hypothesised mean is equal to the sample mean)}
    \item{lam}{Vector of Lagrange multipliers}
    \item{wts}{Observation weights/probabilities (vector of length n)}
    \item{deriv}{Length-2 vector: directional first and second derivatives along the ray toward mu (if \code{deriv = TRUE})}
    \item{converged}{\code{TRUE} if algorithm converged. \code{FALSE} usually means that mu is not in the convex hull of the data. Then, a very small likelihood is returned (instead of zero).}
    \item{iter}{Number of iterations taken.}
    \item{ndec}{Newton decrement (see Boyd & Vandenberghe).}
    \item{gradnorm}{Norm of the gradient of log empirical likelihood.}
}
}
\description{
Implements the empirical-likelihood-ratio test for the mean of the coordinates of \code{z}
(with the hypothesised value \code{mu}). The counts need not be integer;
in the context of local likelihoods, they can be kernel observation weights.
}
\details{
Negative weights are not allowed. They could be useful in some applications, but they can destroy
convexity or even boundedness. They also make the Newton step fail to be of least squares type.

This function relies on the improved computational strategy for the empirical likelihood.
The search of the lambda multipliers is carried out via a dampened Newton method with guaranteed
convergence owing to the fact that the log-likelihood is replaced by its Taylor approximation
of any desired order (default: 4, the minimum value that ensures self-concordance).

Implementation note: the EL solver also guarantees a descent direction; if the Newton step is non-descent or non-finite,
it falls back to steepest descent (negative gradient), which keeps the line search well-behaved.

Tweak \code{alpha} and \code{beta} with extreme caution. See \insertCite{boyd2004convex}{smoothemplik},
pp. 464--466 for details. It is necessary that \code{0 < alpha < 1/2} and \code{0 < beta < 1}.
\code{alpha = 0.3} seems better than 0.01 on some 2-dimensional test data (sometimes fewer iterations).

The argument names, except for \code{lambda.init}, are matching the original names in Art B. Owen's implementation.
The highly optimised one-dimensional counterpart, [EL0()], is designed to return a faster
and a more accurate solution in the one-dimensional case.
}
\examples{
earth <- c(
  5.5, 5.61, 4.88, 5.07, 5.26, 5.55, 5.36, 5.29, 5.58, 5.65, 5.57, 5.53, 5.62, 5.29,
  5.44, 5.34, 5.79, 5.1, 5.27, 5.39, 5.42, 5.47, 5.63, 5.34, 5.46, 5.3, 5.75, 5.68, 5.85
)
EL1(earth, mu = 5.517, verbose = TRUE) # 5.517 is the modern accepted value

# Linear regression through empirical likelihood
coef.lm <- coef(lm(mpg ~ hp + am, data = mtcars))
xmat <- cbind(1, as.matrix(mtcars[, c("hp", "am")]))
yvec <- mtcars$mpg
foc.lm <- function(par, x, y) {  # The sample average of this
  resid <- y - drop(x \%*\% par)   # must be 0
  resid * x
}
minusEL <- function(par) -EL1(foc.lm(par, xmat, yvec), itermax = 10)$logelr
coef.el <- optim(c(26, -0.06, 5.3), minusEL, control = list(maxit = 100))$par
abs(coef.el - coef.lm) / coef.lm  # Relative difference

# Likelihood ratio testing without any variance estimation
# Define the profile empirical likelihood for the coefficient on am
minusPEL <- function(par.free, par.am)
  -EL1(foc.lm(c(par.free, par.am), xmat, yvec), itermax = 20)$logelr
# Constrained maximisation assuming that the coef on par.am is 3.14
coef.el.constr <- optim(coef.el[1:2], minusPEL, par.am = 3.14)$par
print(-2 * EL1(foc.lm(c(coef.el.constr, 3.14), xmat, yvec))$logelr)
# Exceeds the critical value qchisq(0.95, df = 1)
}
\references{
\insertAllCited{}
}
\seealso{
[logTaylor()], [EL0()]
}
