% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/population_dispersal-functions.R
\name{cellular_automata_dispersal}
\alias{cellular_automata_dispersal}
\title{Cellular automata dispersal}
\usage{
cellular_automata_dispersal(
  max_cells = Inf,
  min_cells = max_cells,
  dispersal_proportion = set_proportion_dispersing(),
  barriers = NULL,
  use_suitability = TRUE,
  carrying_capacity = "carrying_capacity"
)
}
\arguments{
\item{max_cells}{the maximum number of cell movements that each individual in
each life stage can disperse in whole integers.}

\item{min_cells}{the minimum number of cell movements that each individual in
each life stage will disperse in whole integers.}

\item{dispersal_proportion}{a built-in or custom function defining the proportions
of individuals that can disperse in each life stage.}

\item{barriers}{the name of a spatial layer in the landscape object that
contains cell values between 0 (no barrier) and 1 (full barrier) Any
values between 0 and 1 indicate the permeability of the barrier.}

\item{use_suitability}{should habitat suitability be used to control the
likelihood of individuals dispersing into cells? The default is TRUE. Note,
if a barrier map is also provided, the suitability map is multiplied with
the barrier map to generate a permeability map of the landscape.}

\item{carrying_capacity}{the name of a spatial layer in the landscape object
that specifies the carrying capacity in each cell.}
}
\description{
The cellular_automata_dispersal function simulates movements of
individuals using rule-based cell movements. In each cell that has
population, every individual up to a specified proportion of the
total population attempts to move. For each step from a specified minimum up
to a specified maximum number of movements, a weighted draw of four
directions, based on habitat suitability, is made and then the destination
cell is checked for available carrying capacity. If there is carrying capacity
available, the individual moves to the cell, if not, it remains in its current
cell. This is repeated until the maximum number of cell movements is reached.
If no cell is found with available carrying capacity, the individual remains
in the source cell.
}
\details{
This function allows the use of barriers in the landscape to influence
dispersal. The function is computationally efficient, however, because
as individuals are dispersed, performance scales with the population sizes
in each cell across a landscape and the maximum number of cell movements.

The maximum number of cell movements in cellular automata dispersal does not
correspond exactly to the distance decay of a dispersal kernel, since cellular
automata dispersal depends on the permeability of the landscape, and is
interrupted on reaching a cell with available capacity (above the minimum
specified number of cell movements). A heuristic that can be used to determine
a reasonable number of steps from a mean dispersal distance `d` and cell
resolution `res` is: `max_cells = round(2 * (d / (res * 1.25)) ^ 2)`. This
corresponds approximately to the number of cell-steps in an infinite,
homogenous landscape with no early stopping, for which d is the mean
end-to-end dispersal distance of all individuals.

Rather than relying on this value, we recommend that the user experiment with
the \code{max_cells} and \code{min_cells} parameters to find a value such that
the the mean dispersal distance in a reasonably realistic simulation
corresponds with field estimates of mean dispersal distances.
}
\examples{

# Example of cellular automata dispersal where the 2nd and 3rd life stages
# disperse up to a maximum of 100 cells but dispersal is affected by
# barriers (in this case roads). The road rasters have values of 0 for
# large roads (no dispersal across barrier) and 0.5 for smaller roads
# (reduced dispersal across barrier).

\dontrun{
ca_dispersal <- cellular_automata_dispersal(max_cells = c(0, 100, 100), barriers = "roads")

ls <- landscape(population = egk_pop,
                suitability = egk_hab,
                carrying_capacity = egk_k,
                "roads" = egk_road)

pd <- population_dynamics(change = growth(egk_mat),
                          dispersal = ca_dispersal,
                          density_dependence = ceiling_density())

simulation(landscape = ls, population_dynamics = pd, habitat_dynamics = NULL, timesteps = 20)
}
}
