#' Process: Simulating Fake Data
#' 
#' @description
#' This function generates random input parameters for a model based on
#'  user-specified distributions. For example, if the first parameter,
#'  \code{eta}, is set to follow a uniform distribution from 0 to 1, its
#'  values will be randomly sampled from \code{U(0, 1)}.
#'
#' You can also specify parameters to be drawn from a normal distribution. 
#'  For example, \code{
#'    eta = function() { stats::rnorm(n = 1, mean = 0.5, sd = 0.1) }
#'  }. 
#'  Make sure the last parameter, which typically represents the inverse 
#'  temperature parameter in the soft-max function, is sampled from an 
#'  exponential distribution.
#' 
#' @param data [data.frame] 
#' 
#' This data should include the following mandatory columns: 
#'  \itemize{
#'    \item "sub"
#'    \item "time_line" (e.g., "Block", "Trial")
#'    \item "L_choice"
#'    \item "R_choice"
#'    \item "L_reward"
#'    \item "R_reward"
#'    \item "sub_choose"
#'  }
#'  
#' @param id [vector]
#' 
#' Specifies which subject's data to use. In parameter and model recovery
#'  analyses, the specific subject ID is often irrelevant. Although the
#'  experimental trial order might have some randomness for each subject,
#'  the sequence of reward feedback is typically pseudo-random.
#'
#' The default value for this argument is \code{NULL}. When \code{id = NULL},
#'  the program automatically detects existing subject IDs within the
#'  dataset. It then randomly selects one subject as a sample, and the
#'  parameter and model recovery procedures are performed based on this
#'  selected subject's data.
#'  
#' default: \code{id = NULL}
#'  
#' @param obj_func [function]
#' 
#' The objective function that the optimization algorithm package accepts.
#'  This function must strictly take only one argument, \code{params} (a vector
#'  of model parameters). Its output must be a single numeric value
#'  representing the loss function to be minimized. For more detailed
#'  requirements and examples, please refer to the relevant documentation 
#'  (
#'     \code{\link[binaryRL]{TD}}, 
#'     \code{\link[binaryRL]{RSTD}}, 
#'     \code{\link[binaryRL]{Utility}}
#'  ).
#' 
#' @param n_params [integer] 
#' 
#' The number of free parameters in your model. 
#' 
#' @param n_trials [integer] 
#' 
#' The total number of trials in your experiment.
#' 
#' @param rfun [List]
#' 
#' A nested list of functions used to generate random parameter values for
#'  simulation. The top-level elements of the list should be named
#'  according to the models. Each of these elements must be a named list
#'  of functions, where each name corresponds to a model parameter and its
#'  value is the random number generation function.
#' 
#' e.g., \code{stats::runif}, \code{stats::rexp}
#'  
#' @param iteration [integer]
#' 
#' This parameter determines how many simulated datasets are created for 
#'  subsequent model and parameter recovery analyses.
#'  
#' default: \code{iteration_s = 10}
#'  
#' @param seed [integer] 
#' 
#' Random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  
#' default: \code{seed = 123}
#'
#' @returns a list with fake data generated by random free parameters
#' 
#' @examples
#' \dontrun{
#' list_simulated <- binaryRL::simulate_list(
#'   data = binaryRL::Mason_2024_G2,
#'   obj_func = binaryRL::RSTD,
#'   n_params = 3,
#'   n_trials = 360,
#'   rfun = list(
#'     etan = function() { stats::runif(n = 1, min = 0, max = 1) },
#'     etap = function() { stats::runif(n = 1, min = 0, max = 1) },
#'     tau = function() { stats::rexp(n = 1, rate = 1) }
#'   ),
#'   iteration = 10
#' )
#' }
#' 
simulate_list <- function(
  data,
  id = 1,
  n_params, 
  n_trials,
  obj_func, 
  rfun,
  iteration = 10,
  seed = 123
) {
  list_simulated <- list()
  # 检测是都用同一个被试的题目, 还是每次都更换题目
  if (length(id) == 1) {
    id <- rep(id, iteration)
  }
  
  for (i in 1:iteration) {
    params <- c()
    
    # 如果没有设置rfun, 则除了逆温度参数服从Exp(1), 其他服从U(0, 1)
    if (is.null(rfun)){
      for (j in 1:n_params) {
        # 确保每次种子不同
        set.seed(seed + n_params * i + j) 
        if (j == n_params) {
          params[j] <- stats::rexp(n = 1, rate = 1)
        } else {
          # 其他参数服从均匀分布
          params[j] <- stats::runif(n = 1, min = 0, max = 1)
        }
      }
    }
    # 如果设置了rfun, 则根据预设分布生成随机数
    else {
      for (j in 1:n_params) {
        # 确保每次种子不同
        set.seed(seed + n_params * i + j) 
        params[j] <- rfun[[j]]()
      }
    }

    # 创建临时环境
    binaryRL.env <- new.env()
    
    # 给临时环境创建全局变量
    binaryRL.env$mode <- "simulate"
    binaryRL.env$policy <- "on"
    
    binaryRL.env$estimate <- "MLE"
    binaryRL.env$priors <- NULL
    
    binaryRL.env$data <- data
    binaryRL.env$id <- id[i]
    binaryRL.env$n_params <- n_params
    binaryRL.env$n_trials <- n_trials
    
    # 让obj_func的环境绑定在fit_env中
    environment(obj_func) <- binaryRL.env
    
    list_simulated[[i]] <- obj_func(params = params)
    list_simulated[[i]]$input <- params
  }
  
  return(list_simulated)
}
