/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Cpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

static herr_t H5C__prefetched_entry_get_initial_load_size(void *udata_ptr, size_t *image_len_ptr);
static herr_t H5C__prefetched_entry_get_final_load_size(const void *image_ptr, size_t image_len,
                                                        void *udata_ptr, size_t *actual_len_ptr);
static htri_t H5C__prefetched_entry_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5C__prefetched_entry_deserialize(const void *image_ptr, size_t len, void *udata,
                                                bool *dirty_ptr);
static herr_t H5C__prefetched_entry_image_len(const void *thing, size_t *image_len_ptr);
static herr_t H5C__prefetched_entry_pre_serialize(H5F_t *f, void *thing, haddr_t addr, size_t len,
                                                  haddr_t *new_addr_ptr, size_t *new_len_ptr,
                                                  unsigned *flags_ptr);
static herr_t H5C__prefetched_entry_serialize(const H5F_t *f, void *image_ptr, size_t len, void *thing);
static herr_t H5C__prefetched_entry_notify(H5C_notify_action_t action, void *thing);
static herr_t H5C__prefetched_entry_free_icr(void *thing);
static herr_t H5C__prefetched_entry_fsf_size(const void *thing, hsize_t *fsf_size_ptr);

H5FL_EXTERN(H5C_cache_entry_t);

const H5AC_class_t H5AC_PREFETCHED_ENTRY[1] = {{
     H5AC_PREFETCHED_ENTRY_ID,
     "prefetched entry",
     H5FD_MEM_DEFAULT, 
     H5AC__CLASS_NO_FLAGS_SET,
     H5C__prefetched_entry_get_initial_load_size,
     H5C__prefetched_entry_get_final_load_size,
     H5C__prefetched_entry_verify_chksum,
     H5C__prefetched_entry_deserialize,
     H5C__prefetched_entry_image_len,
     H5C__prefetched_entry_pre_serialize,
     H5C__prefetched_entry_serialize,
     H5C__prefetched_entry_notify,
     H5C__prefetched_entry_free_icr,
     H5C__prefetched_entry_fsf_size,
}};

static herr_t
H5C__prefetched_entry_get_initial_load_size(void H5_ATTR_UNUSED   *udata_ptr,
                                            size_t H5_ATTR_UNUSED *image_len_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5C__prefetched_entry_get_final_load_size(const void H5_ATTR_UNUSED *image_ptr,
                                          size_t H5_ATTR_UNUSED image_len, void H5_ATTR_UNUSED *udata_ptr,
                                          size_t H5_ATTR_UNUSED *actual_len_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static htri_t
H5C__prefetched_entry_verify_chksum(const void H5_ATTR_UNUSED *image_ptr, size_t H5_ATTR_UNUSED len,
                                    void H5_ATTR_UNUSED *udata_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static void *
H5C__prefetched_entry_deserialize(const void H5_ATTR_UNUSED *image_ptr, size_t H5_ATTR_UNUSED len,
                                  void H5_ATTR_UNUSED *udata, bool H5_ATTR_UNUSED *dirty_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(NULL)
} 

static herr_t
H5C__prefetched_entry_image_len(const void H5_ATTR_UNUSED *thing, size_t H5_ATTR_UNUSED *image_len_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5C__prefetched_entry_pre_serialize(H5F_t H5_ATTR_UNUSED *f, void H5_ATTR_UNUSED *thing,
                                    haddr_t H5_ATTR_UNUSED addr, size_t H5_ATTR_UNUSED len,
                                    haddr_t H5_ATTR_UNUSED *new_addr_ptr, size_t H5_ATTR_UNUSED *new_len_ptr,
                                    unsigned H5_ATTR_UNUSED *flags_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5C__prefetched_entry_serialize(const H5F_t H5_ATTR_UNUSED *f, void H5_ATTR_UNUSED *image_ptr,
                                size_t H5_ATTR_UNUSED len, void H5_ATTR_UNUSED *thing)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5C__prefetched_entry_notify(H5C_notify_action_t action, void *_thing)
{
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)_thing;
    unsigned           u;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(entry_ptr);
    assert(entry_ptr->prefetched);

    switch (action) {
        case H5C_NOTIFY_ACTION_AFTER_INSERT:
        case H5C_NOTIFY_ACTION_AFTER_LOAD:
        case H5C_NOTIFY_ACTION_AFTER_FLUSH:
        case H5C_NOTIFY_ACTION_ENTRY_DIRTIED:
        case H5C_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5C_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5C_NOTIFY_ACTION_CHILD_CLEANED:
        case H5C_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5C_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        case H5C_NOTIFY_ACTION_BEFORE_EVICT:
            for (u = entry_ptr->flush_dep_nparents - 1; u < entry_ptr->flush_dep_nparents; u--) {
                H5C_cache_entry_t *parent_ptr;

                
                assert(entry_ptr->flush_dep_parent);
                parent_ptr = entry_ptr->flush_dep_parent[u];
                assert(parent_ptr);
                assert(parent_ptr->flush_dep_nchildren > 0);

                
                if (H5C_destroy_flush_dependency(parent_ptr, entry_ptr) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                                "unable to destroy prefetched entry flush dependency");

                if (parent_ptr->prefetched) {
                    
                    assert(parent_ptr->fd_child_count > 0);
                    (parent_ptr->fd_child_count)--;
                } 
            }     
            break;

        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__prefetched_entry_free_icr(void *_thing)
{
    H5C_cache_entry_t *entry_ptr = (H5C_cache_entry_t *)_thing;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(entry_ptr);
    assert(entry_ptr->prefetched);

    
    if (entry_ptr->fd_parent_addrs != NULL) {
        assert(entry_ptr->fd_parent_count > 0);
        entry_ptr->fd_parent_addrs = (haddr_t *)H5MM_xfree((void *)entry_ptr->fd_parent_addrs);
    } 
    else
        assert(entry_ptr->fd_parent_count == 0);

    if (entry_ptr->image_ptr != NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "prefetched entry image buffer still attached?");

    entry_ptr = H5FL_FREE(H5C_cache_entry_t, entry_ptr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__prefetched_entry_fsf_size(const void H5_ATTR_UNUSED *thing, hsize_t H5_ATTR_UNUSED *fsf_size_ptr)
{
    FUNC_ENTER_PACKAGE_NOERR 

        HERROR(H5E_CACHE, H5E_SYSTEM, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 
