% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/import.R
\name{import}
\alias{import}
\title{Import m-Path Sense files into a database}
\usage{
import(
  path = getwd(),
  db,
  sensors = NULL,
  batch_size = 24,
  backend = "RSQLite",
  recursive = TRUE
)
}
\arguments{
\item{path}{The path to the file directory}

\item{db}{Valid database connection, typically created by \code{\link[=create_db]{create_db()}}.}

\item{sensors}{Select one or multiple sensors as in \code{\link[mpathsenser]{sensors}}. Leave
NULL to extract all sensor data.}

\item{batch_size}{The number of files that are to be processed in a single batch.}

\item{backend}{Name of the database backend that is used. Currently, only RSQLite is supported.}

\item{recursive}{Should the listing recurse into directories?}
}
\value{
A message indicating how many files were imported. If all files were imported
successfully, this functions returns an empty string invisibly. Otherwise the file names of the
files that were not imported are returned visibly.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Import JSON files from m-Path Sense into a structured database. This function is the bread and
butter of this package, as it populates the database with data that most of the other functions
in this package use. It is recommend to first run \code{\link[=test_jsons]{test_jsons()}} and, if necessary,
\code{\link[=fix_jsons]{fix_jsons()}} to repair JSON files with problematic syntax.
}
\details{
\code{import} allows you to specify which sensors to import (even though there may be more in
the files) and it also allows batching for a speedier writing process. If processing in
parallel is active, it is recommended that \code{batch_size} be a scalar multiple of the number of
CPU cores the parallel cluster can use. If a single JSON file in the batch causes and error,
the batch is terminated (but not the function) and it is up to the user to fix the file. This
means that if \code{batch_size} is large, many files will not be processed. Set \code{batch_size} to 1
for sequential (one-by-one) file processing.

Currently, only SQLite is supported as a backend. Due to its concurrency restriction, parallel
processing works for cleaning the raw data, but not for importing it into the database. This is
because SQLite does not allow multiple processes to write to the same database at the same
time. This is a limitation of SQLite and not of this package. However, while files are
processing individually (and in parallel if specified), writing to the database happens for the
entire batch specified by \code{batch_size} at once. This means that if a single file in the batch
causes an error, the entire batch is skipped. This is to ensure that the database is not left
in an inconsistent state.
}
\section{Parallel}{
 This function supports parallel processing in the sense that it is able to
distribute it's computation load among multiple workers. To make use of this functionality, run
\href{https://rdrr.io/cran/future/man/plan.html}{\code{future::plan("multisession")}} before
calling this function.
}

\section{Progress}{
 You can be updated of the progress of this function by using the
\code{\link[progressr:progress]{progressr::progress()}} package. See \code{progressr}'s
\href{https://cran.r-project.org/package=progressr/vignettes/progressr-intro.html}{vignette} on
how to subscribe to these updates.
}

\examples{
\dontrun{
path <- "some/path"
# Create a database
db <- create_db(path = path, db_name = "my_db")

# Import all JSON files in the current directory
import(path = path, db = db)

# Import all JSON files in the current directory, but do so sequentially
import(path = path, db = db, batch_size = 1)

# Import all JSON files in the current directory, but only the accelerometer data
import(path = path, db = db, sensors = "accelerometer")

# Import all JSON files in the current directory, but only the accelerometer and gyroscope data
import(path = path, db = db, sensors = c("accelerometer", "gyroscope"))

# Remember to close the database
close_db(db)
}

}
\seealso{
\code{\link[=create_db]{create_db()}} for creating a database for \code{import()} to use, \code{\link[=close_db]{close_db()}} for closing
this database; \code{\link[=index_db]{index_db()}} to create indices on the database for faster future processing, and
\code{\link[=vacuum_db]{vacuum_db()}} to shrink the database to its minimal size.
}
