\name{pathmox}
\alias{pathmox}
\title{PATHMOX Algorithm}
\description{
  PATHMOX Approach: Segmentation Trees in Partial Least Squares Path Modeling
}
\usage{
  pathmox(pls, EXEV, X = NULL, signif = 0.05, size = 0.1, deep = 2, tree = TRUE)
}
\arguments{
  \item{pls}{An object of class \code{"plspm"} returned by \code{\link{plspm}}.}
  \item{EXEV}{A data frame of factors contaning the segmentation variables.}
  \item{X}{Optional dataset (matrix or data frame) used when argument \code{dataset=NULL} inside \code{pls}.}
  \item{signif}{A numeric value indicating the significance threshold of the F-statistic. Must be a decimal number between 0 and 1.}
  \item{size}{A numeric value indicating the minimum size of elements inside a node.}
  \item{deep}{An integer indicating the depth level of the tree. Must be an integer greater than 1.}
  \item{tree}{A logical value indicating if the tree should be displayed (\code{TRUE} by default).}
}
\details{
  The function \code{pathmox} calculates a binary segmentation tree following the PATHMOX algorithm. 

  The argument \code{EXEV} must be a data frame containing segmentation variables as factors (see \code{\link{factor}}). The number of rows in \code{EXEV} must be the same as the number of rows in the data used in \code{pls}. \cr

  The argument \code{size} can be defined as a decimal value (i.e. proportion of elements inside a node), or as an integer (i.e. number of elements inside a node). \cr

  When the object \code{pls} does not contain a data matrix (i.e. \code{pls$data=NULL}), the user must provide the data matrix or data frame in \code{X}.
}
\value{
  An object of class \code{"treemox"}. Basically a list with the following results:
  \item{MOX}{Data frame containing the results of the segmentation tree.}
  \item{FT}{Data frame containing the results of the F-test for each node partition.}
  \item{candidates}{List of data frames containing the candidate splits of each node partition.}
  \item{list.nodes}{List of elements for each node.}
}
\references{
  Sanchez, G. (2009) \emph{PATHMOX Approach: Segmentation Trees in Partial Least Squares Path Modeling.} Doctoral Dissertation. Departament Estadistica i Investigacio Operativa, Universitat Politecnica de Catalunya.
}
\author{
  Gaston Sanchez
}
\seealso{\code{\link{techmox}}, \code{\link{plot.treemox}}, \code{\link{treemox.pls}}, \code{\link{plspmox}}.}
\examples{
  \dontrun{
  ## example of PLS-PM in customer satisfaction analysis
  ## model with seven LVs and reflective indicators
  data(csimobile)
  data.mobile <- csimobile[,8:33]  ## select the manifest variables
  IMAG <- c(0,0,0,0,0,0,0)
  EXPE <- c(1,0,0,0,0,0,0)
  QUAL <- c(0,1,0,0,0,0,0)
  VAL <- c(0,1,1,0,0,0,0)
  SAT <- c(1,1,1,1,0,0,0)
  COM <- c(0,0,0,0,1,0,0)
  LOY <- c(1,0,0,0,1,1,0)
  inner.mob <- rbind(IMAG, EXPE, QUAL, VAL, SAT, COM, LOY)
  outer.mob <- list(1:5,6:9,10:15,16:18,19:21,22:24,25:26)
  modes.mob <- rep("A",7)
  pls.mob <- plspm(data.mobile, inner.mob, outer.mob, modes.mob, scheme="factor", scaled=FALSE)

  # re-ordering those segmentation variables with ordinal scale (Age and Education)
  csimobile$Education <- factor(csimobile$Education, 
    levels=c("basic","highschool","university"),
    ordered=TRUE)
  # select the segmentation variables
  EXEV.mob = csimobile[,1:7]
  # Pathmox Analysis
  mob.pathmox <- pathmox(pls.mob, EXEV.mob, signif=.10, size=.10, deep=2)


  ## example of PLS-PM in customer satisfaction analysis
  ## model with six LVs and reflective indicators
  data(csibank)
  data.bank <- csibank[,6:32]  ## select the manifest variables
  IMAG <- c(0,0,0,0,0,0)
  EXPE <- c(1,0,0,0,0,0)
  QUAL <- c(0,1,0,0,0,0)
  VAL  <- c(0,1,1,0,0,0)
  SAT  <- c(1,1,1,1,0,0) 
  LOY  <- c(1,0,0,0,1,0)
  inner.bank <- rbind(IMAG, EXPE, QUAL, VAL, SAT, LOY)
  outer.bank <- list(1:6,7:10,11:17,18:21,22:24,25:27)
  modes.bank <- rep("A", 6)   ## reflective indicators
  pls.bank <- plspm(data.bank, inner.bank, outer.bank, modes.bank, scaled=FALSE)
  
  # re-ordering those segmentation variables with ordinal scale (Age and Education)
  csibank$Age <- factor(csibank$Age, 
      levels=c("<=25","26-35","36-45","46-55","56-65",">=66"),
      ordered=TRUE)  
  csibank$Education <- factor(csibank$Education, 
      levels=c("Unfinished","Elementary","Highschool","Undergrad","Graduated"),
      ordered=TRUE)
  # select the segmentation variables
  EXEV.bank = csibank[,1:5]
  # Pathmox Analysis
  bank.pathmox <- pathmox(pls.bank, EXEV.bank, signif=.05, size=.10, deep=3)
  }
}

