\name{diagnosticPlot}
\alias{diagnosticPlot}
\alias{diagnosticPlot.default}
\alias{diagnosticPlot.optSparseLTSGrid}
\alias{diagnosticPlot.sparseLTS}
\alias{diagnosticPlot.sparseLTSGrid}
\alias{plot.sparseLTS}
\title{Diagnostic plots for sparse LTS regression models}
\usage{
  diagnosticPlot(x, ...)

  \method{diagnosticPlot}{sparseLTS} (x,
    fit = c("reweighted", "raw", "both"), ...)

  \method{diagnosticPlot}{sparseLTSGrid} (x, s,
    fit = c("reweighted", "raw", "both"), ...)

  \method{diagnosticPlot}{optSparseLTSGrid} (x,
    fit = c("reweighted", "raw", "both"), ...)

  \method{diagnosticPlot}{default} (x,
    which = c("all", "rqq", "rindex", "rfit", "rdiag"),
    ask = (which == "all"), facets = attr(x, "facets"),
    size = c(2, 4), id.n = NULL, ...)

  \method{plot}{sparseLTS} (x, ...)
}
\arguments{
  \item{x}{the model fit for which to produce diagnostic
  plots, or a data frame containing all necessary
  information for plotting (as generated by the
  \code{\link[=fortify.sparseLTS]{fortify}} method).}

  \item{s}{an integer vector giving the indices of the
  models for which to produce diagnostic plots.  The
  default is to use the optimal model for each of the
  requested fits.}

  \item{fit}{a character string specifying for which fit to
  produce diagnostic plots.  Possible values are
  \code{"reweighted"} (the default) for diagnostic plots
  for the reweighted fit, \code{"raw"} for diagnostic plots
  for the raw fit, or \code{"both"} for diagnostic plots
  for both fits.}

  \item{which}{a character string indicating which plot to
  show.  Possible values are \code{"all"} (the default) for
  all of the following, \code{"rqq"} for a normal Q-Q plot
  of the standardized residuals, \code{"rindex"} for a plot
  of the standardized residuals versus their index,
  \code{"rfit"} for a plot of the standardized residuals
  versus the fitted values, or \code{"rdiag"} for a
  regression diagnostic plot (standardized residuals versus
  robust Mahalanobis distances of the predictor
  variables).}

  \item{ask}{a logical indicating whether the user should
  be asked before each plot (see
  \code{\link[grDevices]{devAskNewPage}}). The default is
  to ask if all plots are requested and not ask otherwise.}

  \item{facets}{a faceting formula to override the default
  behavior.  If supplied, \code{\link[ggplot2]{facet_wrap}}
  or \code{\link[ggplot2]{facet_grid}} is called depending
  on whether the formula is one-sided or two-sided.}

  \item{size}{a numeric vector of length two giving the
  point and label size, respectively.}

  \item{id.n}{an integer giving the number of the most
  extreme observations to be identified by a label.  The
  default is to use the number of identified outliers,
  which can be different for the different plots.  See
  \dQuote{Details} for more information.}

  \item{\dots}{for the generic function
  \code{diagnosticPlot}, additional arguments to be passed
  down to methods.  For the \code{"sparseLTS"},
  \code{"sparseLTSGrid"} and \code{"optSparseLTSGrid"}
  methods of \code{diagnosticPlot}, additional arguments to
  be passed down to the default method.  For the default
  method of \code{diagnosticPlot}, additional arguments to
  be passed down to \code{\link[ggplot2]{geom_point}}.  For
  the \code{"sparseLTS"} method of \code{plot}, additional
  arguments to be passed down to \code{diagnosticPlot}.}
}
\value{
  If only one plot is requested, an object of class
  \code{"ggplot"} (see \code{\link[ggplot2]{ggplot}}),
  otherwise a list of such objects.
}
\description{
  Produce diagnostic plots for sparse least trimmed squares
  regression models.  Four plots are currently implemented.
}
\details{
  In the normal Q-Q plot of the standardized residuals, a
  reference line is drawn through the first and third
  quartile.  The \code{id.n} observations with the largest
  distances from that line are identified by a label (the
  observation number).  The default for \code{id.n} is the
  number of regression outliers, i.e., the number of
  observations with outlier weight equal to 0 (see
  \code{\link[=wt.sparseLTS]{wt}}).

  In the plots of the standardized residuals versus their
  index or the fitted values, horizontal reference lines
  are drawn at 0 and +/-2.5.  The \code{id.n} observations
  with the largest absolute values of the standardized
  residuals are identified by a label (the observation
  number).  The default for \code{id.n} is the number of
  regression outliers, i.e., the number of observations
  with outlier weight equal to 0 (see
  \code{\link[=wt.sparseLTS]{wt}}).

  For the regression diagnostic plot, the robust
  Mahalanobis distances of the predictor variables are
  computed via the MCD based on only those predictors with
  non-zero coefficients (see
  \code{\link[robustbase]{covMcd}}).  Horizontal reference
  lines are drawn at +/-2.5 and a vertical reference line
  is drawn at the upper 97.5\% quantile of the
  \eqn{\chi^{2}}{chi-squared} distribution with \eqn{p}
  degrees of freedom, where \eqn{p} denotes the number of
  predictors with non-zero coefficients.  The \code{id.n}
  observations with the largest absolute values of the
  standardized residuals and/or largest robust Mahalanobis
  distances are identified by a label (the observation
  number).  The default for \code{id.n} is the number of
  all outliers: regression outliers (i.e., observations
  with outlier weight equal to 0, see
  \code{\link[=wt.sparseLTS]{wt}}) and leverage points
  (i.e., observations with robust Mahalanobis distance
  larger than the 97.5\% quantile of the
  \eqn{\chi^{2}}{chi-squared} distribution with \eqn{p}
  degrees of freedom).
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points

## sparse LTS
# fit model
fit <- sparseLTS(x, y, lambda = 0.05, mode = "fraction")
# create plot
plot(fit)
plot(fit, fit = "both")

## sparse LTS over a grid of values for lambda
# fit model
frac <- seq(0.25, 0.05, by = -0.05)
fitGrid <- sparseLTSGrid(x, y, lambda = frac, mode = "fraction")
# create plot
diagnosticPlot(fitGrid)
diagnosticPlot(fitGrid, fit = "both")
}
\author{
  Andreas Alfons
}
\seealso{
  \code{\link[ggplot2]{ggplot}},
  \code{\link[robustbase:ltsPlot]{plot.lts}},
  \code{\link{sparseLTS}}, \code{\link{sparseLTSGrid}}
}
\keyword{hplot}

