% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/task_spectral_indic.R,
%   R/task_spectral_indic_indictest.R, R/task_spectral_indic_plot.R
\name{spectral_sews}
\alias{spectral_sews}
\alias{indictest.spectral_sews}
\alias{plot.spectral_sews_test}
\alias{plot_spectrum}
\title{Spectrum-based spatial early-warning signals.}
\usage{
spectral_sews(mat, sdr_low_range = NULL, sdr_high_range = NULL,
  quiet = FALSE)

\method{indictest}{spectral_sews}(x, nperm = 999, ...)

\method{plot}{spectral_sews_test}(x, ..., along = NULL, what = "value",
  display_null = TRUE)

plot_spectrum(x, along = NULL, display_null = TRUE)
}
\arguments{
\item{mat}{The input matrix or a list of matrices.}

\item{sdr_low_range}{The range of values (in proportion) to 
use for the computation of the spectral density ratio.
For example, for the lowest 20% (default value), set sdr_low_range to 
c(0, .2). See also the Details section.}

\item{sdr_high_range}{The range of values (in proportion) to 
use for the computation of the spectral density ratio. For example, for 
the higher 20% (default value), set sdr_high_range to 
c(.8, 1). See also the Details section.}

\item{quiet}{Do not display some warnings}

\item{x}{A \code{spectral_sews_test} object as produced by \link{indictest}}

\item{nperm}{The number of replicates to use to compute use in the 
null distribution}

\item{...}{Ignored}

\item{along}{An optional vector of values along which the trend is to be 
displayed. The length of the vector must be equal to the number of 
indicator values provided in the object \code{x}.}

\item{what}{What the trend-line to be displayed should represent about the 
indicator. Defaults to the indicator's values ("value") but other metrics 
can be displayed. Correct values are "value", "pval" or "z_score".}

\item{display_null}{Sets whether a grey ribbon should be added to reflect
the null distribution. Note that it can not be displayed when the trend 
line reflects something else than the indicator values (when \code{what} 
is not set to "value").}
}
\value{
Function \code{spectral_sews} object of class \code{spectral_sews_list} or 
  \code{spectral_sews_single} depending on whether the input was a list of 
  matrices or a single matrix. 

Function \code{indictest} 

The \code{plot} methods returns a ggplot object (usually displayed 
immediately when called interactively).
}
\description{
Computation of spatial early warning signals based on spectral
  properties.
}
\details{
Spectral early warning signals are based on the fact that some 
  dynamical systems can exhibit an change in some characteristics of their 
  spatial structure when approaching a transition. In particular, long-range 
  correlations are expected to have an increased importance. 

This is expected to be reflected in the spectrum of the spatial structure
  by an increase of the relative importance of lower frequencies over higher 
  frequencies ("reddening" of the spectrum).
  
This task allows computing the radial-spectrum which gives the relative 
  importance of each space scale as a function of distance, from 1 to 
  \code{N/2} (N being the minimum between the number of rows and columns). 
  If the matrix is not square, then it is cropped to biggest square that 
  fits within the left side of the matrix. 

Additionally, it summarizes this spectrum into a Spectral 
  Density Ratio (SDR), which is the ratio of low frequencies over 
  high frequencies of the r-spectrum. The SDR value is expected to increase
  before a transition point.

The significance of spectral early-warning signals can be estimated by 
  reshuffling the original matrix (function \code{indictest}). Indicators 
  are then recomputed on the shuffled matrices and the values obtained are 
  used as a null distribution. P-values are obtained based on the rank of 
  the observered value in the null distribution. 

The trend of SDR values can be plotted using the \code{plot()} method. 
  Alternatively, the spectrum itself can be plotted (with facets 
  if multiple input matrices were used) using the \code{plot_spectrum} 
  method.
}
\examples{

data(serengeti) 
data(serengeti.rain) 


spec_indic <- spectral_sews(serengeti, 
                             sdr_low_range  = c(0, .2), 
                             sdr_high_range = c(.8, 1))

summary(spec_indic)

# Display trends along the varying model parameter
plot(spec_indic, along = serengeti.rain)

# Computing spectra many times is expensive, consider setting parallel 
# computing using: options(mc.cores = n)

# Assess significance
\dontrun{
spec_test <- indictest(spec_indic)
summary(spec_test)

# Display the SDR trend, now with a grey ribbon representing 5\%-95\% 
# quantiles of the null distribution
plot(spec_test, along = serengeti.rain)

# Add a line highlighting the shift 
if (require(ggplot2)) {
  plot(spec_test, along = serengeti.rain) + 
    geom_vline(xintercept = 590, color = "red", linetype = "dashed")
}

}

# Display radial-spectra
plot_spectrum(spec_indic, along = serengeti.rain)

# Graphics can be modified using ggplot2 functions
if (require(ggplot2)) { 
  plot_spectrum(spec_indic, along = serengeti.rain) + 
    scale_y_log10()
}

}
\references{
Kefi, S., Guttal, V., Brock, W.A., Carpenter, S.R., Ellison, A.M., 
  Livina, V.N., et al. (2014). Early Warning Signals of Ecological 
  Transitions: Methods for Spatial Patterns. PLoS ONE, 9, e92097.
}
