\name{ptwise.envelope}
\alias{ptwise.envelope}
\alias{bias.envelope}
\alias{RMSE.envelope}
\title{Pointwise Statistics on an Envelope Object}
\description{
  Compute pointwise statistics from the simulated
  function values in an envelope object.
}
\usage{
ptwise.envelope(object,
                stats=c("mean", "median", "bias",
                        "var", "sd", "se", "mse", "rmse",
                        "confint", "predint"),
               \dots,
               level=0.95, transform=NULL, theo=NULL)

bias.envelope(object, theo, CI=TRUE, level=0.95)

RMSE.envelope(object, theo)
}
\arguments{
  \item{object}{An object of class \code{"envelope"}
    generated by the function \code{\link[spatstat.explore]{envelope}}
  }
  \item{stats}{
    Summary statistic(s) to be calculated. A character string
    or character vector (partially matched) selected from the
    options given, or a \code{function(x)} provided by the user
    to compute the summary statistic.
  }
  \item{\dots}{
    Arguments passed to \code{stats} if it is a function.
  }
  \item{level}{
    Confidence level required for \code{confint} or \code{predint}.
    A probability value between 0 and 1.
  }
  \item{transform}{
    Optional expression (passed to
    \code{\link[spatstat.explore]{with.fv}})
    which should be applied to the function values
    before the summary statistics are calculated.
  }
  \item{theo}{
    Function in the \R language that evaluates the
    true (theoretically expected) value of the
    spatial summary function. This is required
    if \code{stats} includes \code{"bias"}, \code{"mse"}
    or \code{"rmse"}.
  }
  \item{CI}{
    Logical value specifying whether to calculate confidence interval
    as well as bias.
  }
}
\details{
  These functions compute pointwise summary statistics
  from \eqn{n} spatial summary functions which were
  obtained from \eqn{n} simulated point patterns.

  The \code{object} should have been generated by the
  function \code{\link[spatstat.explore]{envelope}}
  with the argument \code{savefuns=TRUE}
  specified.

  The function \code{\link[spatstat.explore]{envelope}} is normally used to
  generate simulation envelopes for a particular spatial summary function,
  such as the \eqn{K} function, by simulating \eqn{n} realisations
  of Complete Spatial Randomness or another model.
  However, when \code{\link[spatstat.explore]{envelope}} is called with
  the argument \code{savefuns=TRUE}, it returns all the individual
  summary functions for the \eqn{n} simulated point patterns.

  These individual functions are extracted by \code{ptwise.envelope}
  which then computes the desired summary statistics.
  The argument \code{stats} specifies the desired summary statistics.
  It can be a character string, or vector of character strings,
  containing any of the following (partially matched):
  \describe{
    \item{mean}{the pointwise sample mean of the functions}
    \item{median}{the pointwise sample median of the functions}
    \item{bias}{the pointwise bias of the functions}
    \item{var}{the pointwise sample variance of the functions}
    \item{sd}{the pointwise sample standard deviation of the functions}
    \item{se}{the standard error of the pointwise sample mean}
    \item{mse}{the pointwise mean squared error}
    \item{rmse}{the pointwise root-mean-squared error}
    \item{confint}{a confidence interval for the true mean}
    \item{predint}{a prediction interval for the function value}
  }
  For \code{confint} or \code{predint} the argument \code{level}
  specifies the confidence level.
  
  Alternatively, the argument \code{stats} can be a user-specified
  function in the \R language, which computes the summary statistic.
  It should accept a vector argument and return a single numerical value.
  
  The result is an object of class \code{"fv"} that can be plotted
  directly. See the Examples for different styles of plot.

  The functions \code{bias.envelope} and \code{RMSE.envelope} are
  wrappers for \code{ptwise.envelope} which calculate the bias
  and root-mean-square error respectively.
}
\value{
  A function value table (object of class \code{"fv"}) containing
  some or all of the
  following columns
  \item{r }{Distance argument \eqn{r}}
  \item{mean }{Pointwise sample mean}
  \item{median }{Pointwise sample median}
  \item{bias }{Pointwise estimated bias}
  \item{var }{Pointwise sample variance}
  \item{sd }{Pointwise sample standard deviation}
  \item{se }{Pointwise standard error of pointwise mean}
  \item{mse }{Pointwise estimated mean squared error}
  \item{rmse }{Pointwise estimated root-mean-squared error}
  \item{loci, hici}{Pointwise confidence interval for the true mean}
  \item{lopi, hipi}{Pointwise prediction interval for the function value}
}
\seealso{
  \code{\link{ISB.envelope}},
  \code{\link{IV.envelope}},
  \code{\link{ISE.envelope}},
  \code{\link{MISE.envelope}}.
}
\examples{
    E <- envelope(cells, Kest, nsim=20, savefuns=TRUE)
    plot(ptwise.envelope(E, c("mean", "confint"), level=0.9))
    plot(ptwise.envelope(E, max, na.rm=TRUE))

    ## statistics for L function
    plot(ptwise.envelope(E, c("mean","confint"),
         transform=quote(sqrt(./pi)), level=0.9))

    ## calculate pointwise bias and RMSE
    ## using the theoretical true value K(r) = pi * r^2
    trueK <- function(r) { pi * r^2 }
    plot(bias.envelope(E, theo=trueK))
    plot(RMSE.envelope(E, theo=trueK))
}
\author{\spatstatAuthors, \tilman and \martinH.}
\keyword{spatial}
\keyword{manip}
\keyword{programming}
\keyword{iteration}
