% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCBO.R
\name{mcbo}
\alias{mcbo}
\title{Median Constrained Bucket Order (MCBO)}
\usage{
mcbo(
  X,
  nbuckets,
  wk = NULL,
  ps = TRUE,
  algorithm = "BB",
  itermax = 10,
  np = 10,
  gl = 100,
  ff = 0.4,
  cr = 0.8,
  use_cpp = TRUE
)
}
\arguments{
\item{X}{A N by M data matrix, in which there are N judges and M objects to be judged. 
Each row is a ranking of the objects which are represented by the columns. 
If X contains the rankings observed only once, the argument wk can be used}

\item{nbuckets}{Integer. The number of buckets (tied groups) the consensus ranking 
must contain. Must be between 2 and M-1 (where M is the number of objects)}

\item{wk}{Optional: the frequency of each ranking in the data}

\item{ps}{Logical. If TRUE, displays progress information on screen. Default TRUE}

\item{algorithm}{Character string specifying the algorithm to use. One of:
\itemize{
  \item "BB" - Branch-and-Bound (exact, default). Best for M <= 15
  \item "quick" - Quick algorithm (heuristic). Best for 15 < M <= 50
  \item "decor" - Differential Evolution (metaheuristic). Best for M > 50
}}

\item{itermax}{Integer. Maximum number of iterations for "quick" and "decor" algorithms. 
Default 10}

\item{np}{Integer. Number of population individuals for "decor" algorithm. Default 10}

\item{gl}{Integer. Generations limit for "decor": maximum number of consecutive 
generations without improvement. Default 100}

\item{ff}{Numeric. Scaling rate for mutation in "decor". Must be in [0,1]. Default 0.4}

\item{cr}{Numeric. Crossover range for "decor". Must be in [0,1]. Default 0.8}

\item{use_cpp}{Logical. If TRUE (default), use optimized C++ implementations 
for core functions (combinpmatr, scorematrix, PenaltyBB2, etc.)}
}
\value{
A list containing the following components:
\tabular{lll}{
  Consensus \tab \tab The consensus ranking with exactly nbuckets buckets\cr
  Tau \tab \tab Averaged TauX rank correlation coefficient\cr
  Eltime \tab \tab Elapsed time in seconds
}
}
\description{
Find the median ranking constrained to exactly b buckets (tied groups) according 
to Kemeny's axiomatic approach. This implements the algorithms described in 
D'Ambrosio et al. (2019) for rank aggregation with fixed bucket constraints.
}
\details{
The median constrained bucket order problem finds the ranking that minimizes the 
sum of Kemeny distances to the input rankings, subject to the constraint that 
the solution must have exactly \code{nbuckets} groups of tied items.

This is useful in applications where the output must conform to predetermined 
categories, such as:
\itemize{
  \item Wine quality classifications (e.g., 5 fixed tiers)
  \item Medical triage systems (fixed severity codes)
  \item Educational grading (fixed letter grades: A, B, C, D, F)
}

The search space is restricted to \eqn{Z^{n/b}}, which contains
\deqn{\sum_{b=1}^{n} b! S(n,b)}
rankings, where S(n,b) is the Stirling number of the second kind.

\strong{Algorithm Selection Guidelines:}
\itemize{
  \item \strong{BB}: Exact solution, guaranteed optimal. Use for M <= 15 items
  \item \strong{quick}: Fast heuristic, near-optimal. Use for 15 < M <= 50 items
  \item \strong{decor}: Best for large problems. Use for M > 50 items
}

For stochastic algorithms (quick, decor), consider running multiple times 
(controlled by \code{itermax}) to avoid local optima.
}
\examples{
\dontrun{
# Simple example with 98 judges ranking 5 items into 3 buckets
data(Idea)
RevIdea <- 6 - Idea  # Reverse ranking
CR <- mcbo(RevIdea, nbuckets = 3, algorithm = "BB")
print(CR$Consensus)
print(CR$Tau)

# Large dataset with Quick algorithm
data(EMD)
CR_quick <- mcbo(EMD[,1:15], nbuckets = 5, wk = EMD[,16], 
                 algorithm = "quick", itermax = 20)
}

}
\references{
D'Ambrosio, A., Iorio, C., Staiano, M., and Siciliano, R. (2019). 
Median constrained bucket order rank aggregation. 
Computational Statistics, 34(2), 787-802. 
\doi{10.1007/s00180-018-0858-z}
}
\seealso{
\code{\link{consrank}} for unconstrained consensus ranking

\code{\link{combinpmatr}} for computing the combined input matrix

\code{\link{scorematrix}} for computing score matrices

\code{\link{tau_x}} for TauX correlation coefficient

\code{\link{kemenyd}} for Kemeny distance

\code{\link{stirling2}} for Stirling numbers (bucket combinatorics)
}
\author{
Antonio D'Ambrosio \email{antdambr@unina.it}
}
\keyword{Bucket}
\keyword{Consensus}
\keyword{Kemeny}
\keyword{Median}
\keyword{Rank}
\keyword{aggregation}
\keyword{bucket}
\keyword{constrained}
\keyword{distance}
\keyword{order}
\keyword{ranking}
