% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aliases.R, R/api.R
\name{DDESONN_run}
\alias{DDESONN_run}
\alias{ddesonn_run}
\title{Legacy alias for \code{\link[=ddesonn_run]{ddesonn_run()}}}
\usage{
DDESONN_run(...)

ddesonn_run(
  x,
  y,
  classification_mode = c("binary", "multiclass", "regression"),
  hidden_sizes = c(64, 32),
  seeds = 1L,
  do_ensemble = FALSE,
  num_networks = if (isTRUE(do_ensemble)) 3L else 1L,
  num_temp_iterations = 0L,
  validation = NULL,
  x_valid = NULL,
  y_valid = NULL,
  model_overrides = list(),
  training_overrides = list(),
  temp_overrides = NULL,
  prediction_data = NULL,
  test = NULL,
  x_test = NULL,
  y_test = NULL,
  prediction_type = c("response", "class"),
  aggregate = c("mean", "median", "none"),
  seed_aggregate = c("mean", "median", "none"),
  threshold = NULL,
  output_root = NULL,
  plot_controls = NULL,
  save_models = TRUE,
  verbose = FALSE,
  verboseLow = FALSE,
  debug = FALSE
)
}
\arguments{
\item{...}{Additional arguments passed through to
\code{\link[=ddesonn_run]{ddesonn_run()}}.}

\item{x}{Training features (the training split) as a data frame, matrix, or tibble.}

\item{y}{Training labels/targets (the training split).}

\item{classification_mode}{Overall problem mode. One of \code{"binary"},
\code{"multiclass"}, or \code{"regression"}.}

\item{hidden_sizes}{Integer vector describing the hidden layer widths.}

\item{seeds}{Integer vector of seeds. A separate model (or ensemble stack) is
trained for each seed.}

\item{do_ensemble}{Logical flag selecting the ensemble container modes
(scenarios C/D). When \code{FALSE}, scenarios A/B are executed.}

\item{num_networks}{Number of ensemble members inside each
\code{\link[=ddesonn_model]{ddesonn_model()}} instance.}

\item{num_temp_iterations}{Number of TEMP iterations to run when
\code{do_ensemble = TRUE} (scenario D). Ignored otherwise.}

\item{validation}{Optional validation list with elements \code{x} and \code{y}.}

\item{x_valid}{Optional validation features. Overrides \code{validation$x} when set.}

\item{y_valid}{Optional validation labels. Overrides \code{validation$y} when set.}

\item{model_overrides}{Named list forwarded to \code{\link[=ddesonn_model]{ddesonn_model()}} allowing
custom architectures.}

\item{training_overrides}{Named list forwarded to \code{\link[=ddesonn_fit]{ddesonn_fit()}} for the
main run(s). Any argument accepted by \code{\link[=ddesonn_fit]{ddesonn_fit()}} may be provided here.
Unspecified values fall back to \code{\link[=ddesonn_training_defaults]{ddesonn_training_defaults()}} for the given
\code{classification_mode} and \code{hidden_sizes}. See \strong{Details} and the example
showing how to inspect defaults.}

\item{temp_overrides}{Optional named list forwarded to \code{\link[=ddesonn_fit]{ddesonn_fit()}} for
TEMP iterations. Defaults to \code{training_overrides}. Use this when TEMP
candidates should train differently than the main model.}

\item{prediction_data}{Optional features for prediction. When supplied,
predictions are computed for each seed/iteration.}

\item{test}{Optional test list with elements \code{x} and \code{y}. When supplied,
the final model computes test metrics (loss and, for classification,
accuracy) and stores them in \code{result$test_metrics}. The run history
(\code{result$history}) mirrors the training metadata (train/validation losses)
and appends \code{test_loss} when test data is provided.}

\item{x_test}{Optional test features. Overrides \code{test$x} when set.}

\item{y_test}{Optional test labels. Overrides \code{test$y} when set.}

\item{prediction_type}{Passed to \code{\link[=ddesonn_predict]{ddesonn_predict()}}.}

\item{aggregate}{Aggregation strategy within a single model (across ensemble
members).}

\item{seed_aggregate}{Aggregation strategy across seeds. Set to \code{"none"} to
keep per-seed prediction matrices.}

\item{threshold}{Optional threshold override for classification prediction.}

\item{output_root}{Optional directory where legacy-style artifacts are
written. When \code{NULL} (default) no files are created.}

\item{plot_controls}{Optional list passed through to \code{\link[=ddesonn_fit]{ddesonn_fit()}} as
\code{plot_controls}. Use this to enable/disable specific report plots or
diagnostics (for example, evaluation report settings). The supported
structure is defined by \code{\link[=ddesonn_fit]{ddesonn_fit()}}; this function does not create
defaults.}

\item{save_models}{Logical; if \code{TRUE} (default) individual models are
persisted when \code{output_root} is supplied.}

\item{verbose}{Logical; emit detailed progress output when TRUE.}

\item{verboseLow}{Logical; emit important progress output when TRUE.}

\item{debug}{Logical; emit debug diagnostics when TRUE.}
}
\value{
Same as \code{\link[=ddesonn_run]{ddesonn_run()}}.

A list (classed as \code{"ddesonn_run_result"}) containing the
configuration, per-seed models, and optional prediction summaries.
}
\description{
This helper re-creates the four orchestration modes that previously lived in
\code{TestDDESONN.R}:
}
\details{
\itemize{
\item Scenario A – single model (\code{do_ensemble = FALSE}, \code{num_networks = 1}).
\item Scenario B – single run with multiple members inside a single model
(\code{do_ensemble = FALSE}, \code{num_networks > 1}).
\item Scenario C – main ensemble container (\code{do_ensemble = TRUE},
\code{num_temp_iterations = 0}).
\item Scenario D – main ensemble plus TEMP iterations (\code{do_ensemble = TRUE},
\code{num_temp_iterations > 0}).
}

The function accepts a training set, optional validation data, and optional
prediction features. It repeatedly instantiates \code{\link[=ddesonn_model]{ddesonn_model()}} objects,
fits them with \code{\link[=ddesonn_fit]{ddesonn_fit()}}, and (when requested) calls
\code{\link[=ddesonn_predict]{ddesonn_predict()}} to surface aggregated predictions.

\strong{Discovering available training overrides}

\code{training_overrides} is a direct pass-through to \code{\link[=ddesonn_fit]{ddesonn_fit()}}. To see the
baseline defaults used by \code{ddesonn_run()}, call:

\code{ddesonn_training_defaults(classification_mode, hidden_sizes)}

To see all tunable training arguments, see \code{?ddesonn_fit}.
}
\examples{
\donttest{
# ============================================================
# DDESONN — FULL example using package data in inst/extdata
# (binary classification; train/valid/test split; scale train-only)
# ============================================================

library(DDESONN)

set.seed(111)

# ------------------------------------------------------------
# 1) Locate package extdata folder (robust across check/install)  
# ------------------------------------------------------------
ext_dir <- system.file("extdata", package = "DDESONN")
if (!nzchar(ext_dir)) {
  stop("Could not find DDESONN extdata folder. Is the package installed?",
       call. = FALSE)
}

# ------------------------------------------------------------
# 1b) Find CSVs (recursive + check-dir edge cases)               
# ------------------------------------------------------------
csvs <- list.files(
  ext_dir,
  pattern = "\\\\\\\\.csv$",
  full.names = TRUE,
  recursive = TRUE
)

# Defensive fallback for rare nested layouts
if (!length(csvs)) {                                             
  ext_dir2 <- file.path(ext_dir, "inst", "extdata")               
  if (dir.exists(ext_dir2)) {                                    
    csvs <- list.files(
      ext_dir2,
      pattern = "\\\\\\\\.csv$",
      full.names = TRUE,
      recursive = TRUE
    )
  }
}

if (!length(csvs)) {
  message(sprintf(
    "No .csv files found under: \%s — skipping example.",
    ext_dir
  ))
} else {

  hf_path <- file.path(ext_dir, "heart_failure_clinical_records.csv")
  data_path <- if (file.exists(hf_path)) hf_path else csvs[[1]]

  cat("[extdata] using:", data_path, "\\\\n")

# ------------------------------------------------------------
# 2) Load data
# ------------------------------------------------------------
df <- read.csv(data_path)

# Prefer DEATH_EVENT if present; otherwise infer a binary target
target_col <- if ("DEATH_EVENT" \%in\% names(df)) {
  "DEATH_EVENT"
} else {
  cand <- names(df)[vapply(df, function(col) {
    v <- suppressWarnings(as.numeric(col))
    if (all(is.na(v))) return(FALSE)
    u <- unique(v[is.finite(v)])
    length(u) <= 2 && all(sort(u) \%in\% c(0, 1))
  }, logical(1))]
  if (!length(cand)) {
    stop(
      "Could not infer a binary target column. ",
      "Provide a binary CSV in extdata or rename target to DEATH_EVENT.",
      call. = FALSE
    )
  }
  cand[[1]]
}

cat("[data] target_col =", target_col, "\\\\n")

# ------------------------------------------------------------
# 3) Build X and y
# ------------------------------------------------------------
y_all <- matrix(as.integer(df[[target_col]]), ncol = 1)

x_df <- df[, setdiff(names(df), target_col), drop = FALSE]
x_all <- as.matrix(x_df)
storage.mode(x_all) <- "double"

# ------------------------------------------------------------
# 4) Split 70 / 15 / 15
# ------------------------------------------------------------
n <- nrow(x_all)
idx <- sample.int(n)

n_train <- floor(0.70 * n)
n_valid <- floor(0.15 * n)

i_tr <- idx[1:n_train]
i_va <- idx[(n_train + 1):(n_train + n_valid)]
i_te <- idx[(n_train + n_valid + 1):n]

x_train <- x_all[i_tr, , drop = FALSE]
y_train <- y_all[i_tr, , drop = FALSE]

x_valid <- x_all[i_va, , drop = FALSE]
y_valid <- y_all[i_va, , drop = FALSE]

x_test  <- x_all[i_te, , drop = FALSE]
y_test  <- y_all[i_te, , drop = FALSE]

cat(sprintf("[split] train=\%d valid=\%d test=\%d\\\\n",
            nrow(x_train), nrow(x_valid), nrow(x_test)))

# ------------------------------------------------------------
# 5) Scale using TRAIN stats only (no leakage)
# ------------------------------------------------------------
x_train_s <- scale(x_train)
ctr <- attr(x_train_s, "scaled:center")
scl <- attr(x_train_s, "scaled:scale")
scl[!is.finite(scl) | scl == 0] <- 1

x_valid_s <- sweep(sweep(x_valid, 2, ctr, "-"), 2, scl, "/")
x_test_s  <- sweep(sweep(x_test,  2, ctr, "-"), 2, scl, "/")

mx <- suppressWarnings(max(abs(x_train_s)))
if (!is.finite(mx) || mx == 0) mx <- 1

x_train <- x_train_s / mx
x_valid <- x_valid_s / mx
x_test  <- x_test_s  / mx

# ------------------------------------------------------------
# 6) Run DDESONN
# ------------------------------------------------------------
res <- ddesonn_run(
  x = x_train,
  y = y_train,
  classification_mode = "binary",

  hidden_sizes = c(64, 32),
  seeds = 1L,
  do_ensemble = FALSE,

  validation = list(
    x = x_valid,
    y = y_valid
  ),

  test = list(
    x = x_test,
    y = y_test
  ),

  training_overrides = list(
    init_method = "he",
    optimizer = "adagrad",
    lr = 0.125,
    lambda = 0.00028,

    activation_functions = list(relu, relu, sigmoid),
    dropout_rates = list(0.10),
    loss_type = "CrossEntropy",

    validation_metrics = TRUE,
    num_epochs = 360,
    final_summary_decimals = 6L
  ),

  plot_controls = list(
    evaluate_report = list(
      roc_curve = TRUE,
      pr_curve  = FALSE
    )
  )
)
}
}
}
