% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hcp_predict_targets.R
\name{hcp_predict_targets}
\alias{hcp_predict_targets}
\title{HCP prediction wrapper for multiple measurements with optional per-patient Bonferroni}
\usage{
hcp_predict_targets(
  dat,
  test,
  pid_col = "pid",
  x_cols,
  y_grid,
  alpha = 0.1,
  bonferroni = FALSE,
  return_region = FALSE,
  id_col = "id",
  y_col = "Y",
  delta_col = "delta",
  ...
)
}
\arguments{
\item{dat}{Training/calibration data passed to \code{\link{hcp_conformal_region}}.}

\item{test}{A data.frame of test measurements, where each row corresponds to a single
measurement. The test data must follow one of the four clustered settings
\eqn{P=1, M=1}, \eqn{P=1, M>1}, \eqn{P>1, M=1}, or \eqn{P>1, M>1}, where \eqn{P} is the
number of patients (clusters) and \eqn{M} is the number of measurements per patient.

The data.frame must include a patient identifier specified by \code{pid_col} and
all covariate columns listed in \code{x_cols}. Repeated values of \code{pid_col}
indicate multiple measurements (e.g., repeated or longitudinal measurements) for
the same patient.}

\item{pid_col}{Column in \code{test} giving the patient (cluster/subject) identifier.
Default \code{"pid"}.}

\item{x_cols}{Covariate column names (e.g., \code{c("X1")}).}

\item{y_grid}{Candidate y-grid passed to \code{\link{hcp_conformal_region}}.}

\item{alpha}{Nominal miscoverage level in (0,1) passed to
\code{\link{hcp_conformal_region}}.}

\item{bonferroni}{Logical; if TRUE, apply per-patient Bonferroni only when a patient
has multiple test measurements (i.e., \eqn{M_p>1}). If FALSE, always use level
\eqn{\alpha}.}

\item{return_region}{Logical; if TRUE, return the full region (subset of
\code{y_grid}) for each row.}

\item{id_col, y_col, delta_col}{Column names in \code{dat} for patient ID, outcome, and
missingness indicator.}

\item{...}{Additional arguments forwarded to
\code{\link{hcp_conformal_region}} (e.g., \code{S}, \code{B}, \code{combine_B},
\code{combine_S}, \code{dens_method}, \code{prop_method}, \code{seed}).}
}
\value{
A list with:
\describe{
\item{pred}{A data.frame in the same row order as \code{test}. It contains all
columns of \code{test} plus the effective level \code{alpha_eff} and the
prediction-band endpoints \code{lo} and \code{hi} for each measurement.}
\item{region}{If \code{return_region=TRUE}, a list of length
\code{nrow(test)} where each element is the subset of \code{y_grid} retained
in the prediction region for the corresponding test row; otherwise
\code{NULL}.}
\item{meta}{A list with summary information, including the number of patients
\code{P}, the per-patient measurement counts \code{M_by_pid}, and the
settings \code{alpha} and \code{bonferroni}.}
}
}
\description{
Wraps \code{\link{hcp_conformal_region}} to produce conformal prediction regions for
a collection of measurements, possibly including multiple measurements per individual.

Based on the structure of the test dataset, the prediction mode is determined
automatically as follows, where \eqn{P} denotes the number of patients (clusters)
and \eqn{M} denotes the number of measurements per patient:
\itemize{
\item \strong{\eqn{P=1,\, M=1}}: Predict a single patient with a single measurement.
\item \strong{\eqn{P=1,\, M>1}}: Predict a single patient with multiple measurements
(e.g., repeated or longitudinal measurements for the same patient). If per-patient
simultaneous prediction is desired, optional per-patient Bonferroni calibration
can be applied.
\item \strong{\eqn{P>1,\, M=1}}: Predict multiple patients, each with a single
measurement. Predictions are performed independently at the nominal level
\eqn{\alpha}, without Bonferroni calibration.
\item \strong{\eqn{P>1,\, M>1}}: Predict multiple patients, each with multiple
measurements. When per-patient simultaneous coverage is desired, a Bonferroni
correction can be applied by using an effective level \eqn{\alpha / M_p} for each
measurement, yielding Bonferroni-adjusted marginal prediction regions for patient
\eqn{p}.
}
}
\note{
When per-patient Bonferroni calibration is enabled and a patient has a large number
of measurements (e.g., \eqn{M_p > 10}), the effective level \eqn{\alpha / M_p} may be
very small, which can lead to extremely wide prediction regions (potentially spanning
the entire \code{y_grid}). This behavior is an inherent consequence of Bonferroni
adjustment and not a numerical issue.

In longitudinal or panel studies, a cluster corresponds to a single individual
(subject), and within-cluster points correspond to multiple time points or repeated
measurements on the same individual. In this setting, the time variable \code{time}
can be treated as a generic covariate. In the examples below, time is represented by
\code{X1}.
}
\examples{
## ------------------------------------------------------------
## Examples illustrating the four test-data settings:
## (P=1, M=1), (P=1, M>1), (P>1, M=1), and (P>1, M>1)
## ------------------------------------------------------------
set.seed(1)

## training data (fixed across all cases)
dat_train <- generate_clustered_mar(
  n = 200, m = 4, d = 1,
  x_dist = "uniform", x_params = list(min = 0, max = 10),
  target_missing = 0.30,
  seed = 1
)

y_grid <- seq(-6, 6, length.out = 201)

## Case 1: P=1, M=1  (one patient, one measurement)
test_11 <- data.frame(
  pid = 1,
  X1  = 2.5
)
out_11 <- hcp_predict_targets(
  dat = dat_train,
  test = test_11,
  x_cols = "X1",
  y_grid = y_grid,
  alpha = 0.1,
  S = 2, B = 2,
  seed = 1
)
out_11$pred

## Case 2: P=1, M>1  (one patient, multiple measurements)
test_1M <- data.frame(
  pid = 1,
  X1  = c(1, 3, 7, 9)
)
out_1M <- hcp_predict_targets(
  dat = dat_train,
  test = test_1M,
  x_cols = "X1",
  y_grid = y_grid,
  alpha = 0.1,
  S = 2, B = 2,
  seed = 1
)
out_1M$pred

## Case 3: P>1, M=1  (multiple patients, one measurement each)
test_P1 <- data.frame(
  pid = 1:4,
  X1  = c(2, 4, 6, 8)
)
out_P1 <- hcp_predict_targets(
  dat = dat_train,
  test = test_P1,
  x_cols = "X1",
  y_grid = y_grid,
  alpha = 0.1,
  S = 2, B = 2,
  seed = 1
)
out_P1$pred

## Case 4: P>1, M>1  (multiple patients, multiple measurements per patient)
test_PM <- data.frame(
  pid = c(1,1, 2,2,2, 3,3),
  X1  = c(1,6,  2,5,9,  3,8)
)
out_PM <- hcp_predict_targets(
  dat = dat_train,
  test = test_PM,
  x_cols = "X1",
  y_grid = y_grid,
  alpha = 0.1,
  S = 2, B = 2,
  seed = 1
)
out_PM$pred
}
