#' Predictor Importance – Variables and Interactions
#'
#' Calculates permutation-based importance measures for individual predictors and interactions
#' within a logic regression tree in a logic forest.
#'
#' @importFrom survival coxph
#'
#' @param fit Fitted logic regression tree object containing outcome, model type, and logic tree information.
#' @param data In-bag sample (training data).
#' @param testdata Out-of-bag sample (test data).
#' @param BSpred Number of predictors included in the interactions (includes NOT-ed variables).
#' @param pred Number of predictors in the model (used for constructing permuted matrices).
#' @param Xs Matrix or data frame of 0/1 values representing all predictor variables.
#' @param mtype Model type: \code{"classification"}, \code{"linear"}, \code{"Cox-PH Time-to-Event"}, or \code{"Exp. Time-to-Event"}.
#'
#' @details
#' This function calculates importance measures for each bootstrapped sample by comparing model fit
#' between the original out-of-bag sample and a permuted out-of-bag sample.
#' Model fit is evaluated using:
#' \itemize{
#'   \item Misclassification rate for classification models,
#'   \item Log2 mean squared error for linear regression,
#'   \item Harrell's C-index for survival regression (Cox-PH or exponential time-to-event models).
#' }
#'
#' @return A list with the following components:
#' \describe{
#'   \item{single.vimp}{Vector of importance estimates for individual predictors.}
#'   \item{pimp.vimp}{Vector of importance estimates for interactions (pimps).}
#'   \item{Ipimat}{Matrix indicating which predictors (and NOT-ed predictors) are used in each interaction.}
#'   \item{vec.Xvars}{Vector of predictor IDs used in the tree.}
#'   \item{Xids}{Vector of predictor column indices corresponding to \code{vec.Xvars}.}
#' }
#'
#' @references
#' Wolf BJ, Hill EG, Slate EH. Logic Forest: an ensemble classifier for discovering logical combinations of binary markers. 
#' \emph{Bioinformatics}. 2010;26(17):2183–2189. \doi{10.1093/bioinformatics/btq354}
#'
#' @author
#' Bethany J. Wolf \email{wolfb@@musc.edu} \cr
#' J. Madison Hyer \email{madison.hyer@@osumc.edu}
#'
#' @seealso \code{\link{logforest}}
#'
#' @keywords internal

pimp.import<-function(fit, data, testdata, BSpred, pred, Xs, mtype)
{
  if (mtype=="Classification") {mtyp=1}
  if (mtype=="Linear Regression") {mtyp=2}
  if (mtype=="Cox-PH Time-to-Event") {mtyp=3}
  if (mtype=="Exp. Time-to-Event") {mtyp=4}

  n <- nrow(testdata) #number obs in testdata (OOB)
  tree <- fit$model$trees[[1]] #gives tree in form of class="tree"
  y <- testdata[,(BSpred+1)]
  OOBY <- y
  if(mtyp %in% 3:4)
  {
    OOBYT <- testdata[,(BSpred+2)]
  }

  x.names <- colnames(data[,1:BSpred])

  #OBTAINS PREDICTED VALUES FOR EACH OBS IN THE OOB SAMPLE
  #THIS BIT IDENTIFIES LINES THE INTERACTION APPLIES TO
  if(mtyp %in% 3:4)
  {
    orig.pred.1 <- frame.logreg2(fit = fit, newbin = testdata[,1:BSpred], newcens = y)
    orig.pred <- orig.pred.1[, 3]
  } else {
    orig.pred <- predict.logreg2(fit, newbin=testdata[,1:BSpred])
  }

  if (mtyp==1) {orig.miss <- sum(abs(orig.pred-y))/n} #misclass rate on test data
  if (mtyp==2) {orig.miss <- log2(sum((orig.pred-y)^2)/n)} #log2MSEP on test data
  if (mtyp==3 | mtyp == 4)
  {
    orig.data <- as.data.frame(cbind(orig.pred, OOBY, OOBYT))
    options(warn = -1)
    cox_model <- coxph(Surv(OOBYT, OOBY) ~ orig.pred, data = orig.data)
    options(warn = 0)
    orig.c <- summary(cox_model)
    orig.miss <- orig.c$concordance[1]
  }

  pimpinfo<-prime.imp(tree=tree, data=data, Xs=Xs, mtype=mtyp) #determines prime imps for tree + tmp matrix

  numorigpis<-nrow(pimpinfo$tmp.mat)
  Xvars <- Xs

  if (mtyp %in% 2:4)
  {
    cpimpinfo <- prime.imp(tree=find.ctree(tree), data=data, Xs=Xvars, mtype=mtyp)
    cloc <- which(rowSums(abs(cpimpinfo$tmp.mat))>=1) #BROKE DOWN WHEN A TREE WAS JUST ONE VAR. CHANGED THIS TO '>=' FROM '>'
    cnewpimps <- cpimpinfo$vec.primes[cloc]
    loc <- which(rowSums(abs(pimpinfo$tmp.mat))>=1) #BROKE DOWN WHEN A TREE WAS JUST ONE VAR. CHANGED THIS TO '>=' FROM '>'
    newpimps <- pimpinfo$vec.primes[loc]
    is.cmp<-rep(c(0, 1), times=c(length(newpimps), length(cnewpimps)))
    pimpinfo$vec.pimpvars <- sort(unique(append(pimpinfo$vec.pimpvars, cpimpinfo$vec.pimpvars))) #ADDING THIS 29MAR
    pimpinfo$vec.primes <- append(newpimps, cnewpimps)
    pimpinfo$tmp.mat <- rbind(pimpinfo$tmp.mat[loc,], cpimpinfo$tmp.mat[cloc,])
    pimpinfo$list.pimps <- append(pimpinfo$list.pimps, cpimpinfo$list.pimps)
    pimpinfo$cmp <- is.cmp
    newvec.primes <- pimpinfo$vec.primes
  }

  Ipimat <- matrix(0, nrow=nrow(pimpinfo$tmp.mat), ncol=pred*2)
  for (i in 1:nrow(pimpinfo$tmp.mat))
  {
    lc <- pimpinfo$tmp.mat[i,][which(pimpinfo$tmp.mat[i,]!=0)]*pimpinfo$list.pimps[[i]]
    lc <- ifelse(lc>0, lc, -1*lc+pred)
    Ipimat[i,lc] <- 1
  }

  colnames(Ipimat)<-c(x.names, paste("!", x.names, sep=""))
  vec.Xvars <- pimpinfo$vec.pimpvars #vector of Xvar ids from tree

  nxvars <- length(vec.Xvars) #number Xvars in tree
  single.vimp <- c()
  single.vimp.nms <- c()
  Xids <- c()
  for (i in 1:nxvars) #checking single var importance for tree
  {
    id<-vec.Xvars[i]
    Xid<-Xs[id]
    permute.ind <- sample(1:n, n, replace = FALSE) #indicators of permuted obs
    permute.col <- testdata[permute.ind, id] #permuted observations for given predictor
    pre.id <- if(id > 1) as.matrix(testdata[,1:(id-1)])
    post.id <- if(id < pred) as.matrix(testdata[,(id+1):BSpred])
    permute.testdata <- cbind(pre.id, permute.col, post.id)

    if(mtyp %in% 3:4)
    {
      perm.pred.1 <- frame.logreg2(fit = fit, newbin = as.matrix(permute.testdata[,1:BSpred]), newcens = y)
      perm.pred <- perm.pred.1[, 3]
    } else {
      perm.pred<-predict.logreg2(fit, newbin=as.matrix(permute.testdata[,1:BSpred]))
    }

    if (mtyp==1) {perm.misclass<-sum(abs(perm.pred-y))/n}   	#number prediction differ bt/ orig & permuted
    if (mtyp==2) {perm.misclass<-log2(sum((perm.pred-y)^2)/n)} #log2MSEP difference bt/ orig & permuted
    if (mtyp==3 | mtyp == 4)
    {
      perm.data <- as.data.frame(cbind(perm.pred, OOBY, OOBYT))
      options(warn = -1)
      cox_model <- coxph(Surv(OOBYT, OOBY) ~ perm.pred, data = perm.data)
      options(warn = 0)
      perm.c <- summary(cox_model)
      perm.misclass <- perm.c$concordance[1]
    }

    if(mtyp %in% 3:4){
      vimp <- orig.miss - perm.misclass
    } else {
      vimp <- perm.misclass - orig.miss
    }
    single.vimp<-append(single.vimp, vimp)
    single.vimp.nms<-append(single.vimp.nms, x.names[id])
    Xids<-append(Xids, Xid)
  }
  names(single.vimp)<-single.vimp.nms

  if(mtyp %in% 2:4)
  {
    pimplist<-pimp.mat.nonbin(pimps.out=pimpinfo, testdata=testdata)
  } else {
    pimplist<-pimp.mat.bin(pimps.out=pimpinfo, testdata=testdata)
  }

  pimpmat <- pimplist[[2]] #transforms data into predictors = pimps
  pimpnames <- pimplist[[1]] #vector of pimp names
  tmp.mat <- pimpinfo$tmp.mat
  zero.ids <- c()
  for(i in 1:ncol(tmp.mat))
  {
    ids <- if(all(tmp.mat[,i]==0)) {ids<-i}
    zero.ids <- append(zero.ids, ids)
  }
  if (length(zero.ids) > 0) {tmp.mat<-tmp.mat[,-zero.ids]}
  if (is.matrix(tmp.mat)) {npimps<-nrow(tmp.mat)}
  if (is.vector(tmp.mat)) {npimps<-1}
  pimp.vimp<-c()
  for (j in 1:npimps) #permuting pimps and checking misclass
  {
    perm.ind<-sample(1:n, n, replace=FALSE)#indicators of premuted obs
    perm.col<-pimpmat[perm.ind, j] #permuted observations for given predictor
    pre.j<-if(j>1) pimpmat[,1:(j-1)]
    post.j<-if(j<npimps) pimpmat[,(j+1):npimps]
    permute.pimpdata<-cbind(pre.j, perm.col, post.j)
    pimp.pred<-c()
    for (k in 1:n)
    {
      if(mtyp==1)
      {
        pred<-ifelse(any(permute.pimpdata[k,]==1), 1, 0)
        pimp.pred<-append(pimp.pred, pred)
      }
      if(mtyp %in% 2:4)
      {
        pred<-ifelse(any(permute.pimpdata[k,]==1), sum(fit$model$coef), fit$model$coef[1])
        pimp.pred<-append(pimp.pred, pred)
      }
    }
    if (mtyp==1) {permpimp.miss<-sum(abs(pimp.pred-y))/n}		#permuted pimp misclass for jth pimp
    if (mtyp==2) {permpimp.miss<-log2(sum((pimp.pred-y)^2)/n)} #permuted pimp log2MSEP for jth pimp
    if (mtyp==3 | mtyp == 4)
    {
      pimp.data <- as.data.frame(cbind(pimp.pred, OOBY, OOBYT))
      options(warn = -1)
      cox_model <- coxph(Surv(OOBYT, OOBY) ~ pimp.pred, data = pimp.data)
      options(warn = 0)
      pimp.c <- summary(cox_model)
      permpimp.miss <- pimp.c$concordance[1]
    }
    if(mtyp %in% 3:4){
      pvimp <- orig.miss - permpimp.miss
    } else {
      pvimp <- permpimp.miss - orig.miss #diff bt/ permutation and original
    }
    pimp.vimp<-append(pimp.vimp, pvimp)
  }
  names(pimp.vimp)<-paste(pimpnames)
  out<-list(single.vimp=single.vimp, pimp.vimp=pimp.vimp, Ipimat=Ipimat, vec.Xvars=vec.Xvars, Xids=Xids)
}
