#' Population Median Absolute Deviation (MAD)
#' @description
#' This function calculates the population Median Absolute Deviation (MAD) for a user-specified distribution.
#'
#' @param qf the quantile function for the distribution.
#' @param pf the distribution function for the distribution.
#' @param upper an optional number specifying an upper limit for the MAD.  This can be missing although the user may want to specify an upper limit if there are computation issues.
#' @param constant a scale factor. Default choice ensures population MAD is equal to SD for the normal distribution.
#' @param ... additional parameter values for the distribution, otherwise default choices are used for the specific quantile and distribution functions.
#'
#' @return A value representing the MAD for the distribution.
#' @details
#' This function calculates the Median Absolute Deviation (MAD) for a distribution specified by the user. The user is required to input the quantile and
#' distribution functions, as well as any parameters required for the nominated distribution.  If the function is unable to find
#' the MAD using function `root`, the user may specify another upper limit for the MAD to help with computation.  If `upper` is missing, `upper`
#' is chosen to be the maximum of the third quartile minus the median and the median minus the first quantile (the MAD cannot be greater than this upper limit).  For more information in the MAD see, e.g., Hampel et al. (1986).
#' @references
#' Hampel, F. R., Ronchetti, E. M., Rousseeuw, P. J. & Stahel, W. A. (1986). Robust statistics: The approach based on influence functions. New York, NY: John Wiley & Sons.
#' @export
#'
#' @examples
#' # Population MAD for the normal distribution
#' pop.mad(qnorm, pnorm, sd = 1.5)
#'
#' # Note that due to the scaling constant used, the mad is equal to the SD for the normal distribution

pop.mad <- function(qf, pf, upper, constant = 1.4826, ...) {
  m <- qf(0.5, ...)

  f <- function(target) {
    pf(m + target, ...) - pf(m - target, ...) - 0.5
  }

  if (missing(upper))
    upper <- max(c(qf(0.75, ...) - m, m - qf(0.25, ...)))

  found.mad <- tryCatch(stats::uniroot(f, c(0, upper))$root,
                        error = function(e){
                          warning(paste("No mad found using function root. ",
                                  "Try setting a value for argument upper: ",
                                  e$message))
                          return(NA)
                        })
  constant*found.mad
}
