# LindleyExpPower.R
#' Lindley-Exponential Power Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley-Exponential Power distribution.
#'
#' The Lindley-Exponential Power distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley-Exponential Power distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad 1 - \left[ {1 - \left( {\frac{\theta }{{1 + \theta }}} \right)
#' \left( {1 - {e^{{{\left( {\lambda x} \right)}^\alpha }}}} \right)} 
#' \right]\exp \left[ {\theta \left( {1 - {e^{{{\left( {\lambda x} \right)}
#' ^\alpha }}}} \right)} \right] \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlind.exp.pow()} — Density function
#'   \item \code{plind.exp.pow()} — Distribution function
#'   \item \code{qlind.exp.pow()} — Quantile function
#'   \item \code{rlind.exp.pow()} — Random generation
#'   \item \code{hlind.exp.pow()} — Hazard function
#' }
#'
#' @usage
#' dlind.exp.pow(x, alpha, lambda, theta, log = FALSE)
#' plind.exp.pow(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlind.exp.pow(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlind.exp.pow(n, alpha, lambda, theta)
#' hlind.exp.pow(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlind.exp.pow}: numeric vector of (log-)densities
#'   \item \code{plind.exp.pow}: numeric vector of probabilities
#'   \item \code{qlind.exp.pow}: numeric vector of quantiles
#'   \item \code{rlind.exp.pow}: numeric vector of random variates
#'   \item \code{hlind.exp.pow}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R. K., & Kumar, V. (2020). 
#' Lindley exponential power distribution with Properties and Applications. 
#' \emph{International Journal for Research in Applied Science & Engineering 
#' Technology (IJRASET)}, \bold{8(10)}, 22--30. \doi{10.22214/ijraset.2020.31743}
#'
#' Joshi, R.K., & Kumar, V. (2016). 
#' Exponentiated Power Lindley Distribution : A Bayes Study using MCMC Approach. 
#' \emph{J. Nat. Acad. Math.}, \bold{30}, 80--102.
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dlind.exp.pow(x, 0.5, 0.2, 1.5)
#' plind.exp.pow(x, 0.5, 0.2, 1.5)
#' qlind.exp.pow(0.5, 0.5, 0.2, 1.5)
#' rlind.exp.pow(10, 0.5, 0.2, 1.5)
#' hlind.exp.pow(x, 0.5, 0.2, 1.5)
#' 
#' # Data
#' x <- windshield
#' # ML estimates
#' params = list(alpha=0.97722, lambda=0.39461, theta=0.96124)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plind.exp.pow, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlind.exp.pow, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlind.exp.pow, pfun=plind.exp.pow, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyExpPower
#' @aliases dlind.exp.pow plind.exp.pow qlind.exp.pow rlind.exp.pow hlind.exp.pow
#' @export
NULL

#' @rdname LindleyExpPower
#' @usage NULL
#' @export   
dlind.exp.pow <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {	
	u <- (lambda * x[valid]) ^ alpha	
	z <- 1 -  exp(u) 	
    pdf[valid] <- (theta*theta/(1+theta)) * alpha * (lambda^alpha) * (x[valid]^(alpha-1)) * exp(u) * exp(theta*z) * (1 - z)   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyExpPower
#' @usage NULL
#' @export 
plind.exp.pow <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	z <- 1 - exp((lambda*q[valid]) ^ alpha)	
    cdf[valid] <- 1.0 - exp(theta * z) * (1 - (theta/(1+theta)) * z)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyExpPower
#' @usage NULL
#' @export 
qlind.exp.pow <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Parameters 'alpha', 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	log(1-(theta/(theta+1))*(1-exp((lambda*x)^alpha)))+theta*(1-exp((lambda*x)^alpha))-log(1-wi)
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- 1e-3
      upper <- 1e3
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-5),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LindleyExpPower
#' @usage NULL
#' @export 
rlind.exp.pow <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qlind.exp.pow(u, alpha, lambda, theta)
    return(q)
}

#' @rdname LindleyExpPower
#' @usage NULL
#' @export 
hlind.exp.pow <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dlind.exp.pow(x, alpha, lambda, theta) 
	surv <- 1.0 - plind.exp.pow(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 




