# NewLindleyHC.R
#' New Lindley Half-Cauchy Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the New Lindley Half-Cauchy distribution.
#'
#' The New Lindley Half-Cauchy distribution is parameterized by the parameters
#' \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The New Lindley Half-Cauchy distribution has CDF:
#'
#' \deqn{
#' F(x; \lambda, \theta) =
#' \left\{\frac{2}{\pi} \tan ^{-1}\left(\frac{x}{\lambda}\right)\right\}
#' ^\theta\left\{1-\left(\frac{\theta}{1+\theta}\right) \ln \left[\frac{2}{\pi} 
#' \tan ^{-1}\left(\frac{x}{\lambda}\right)\right]\right\}  \quad ;\;x > 0.
#' }
#'
#' where\eqn{\lambda} and \eqn{\theta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dNLindley.HC()} — Density function
#'   \item \code{pNLindley.HC()} — Distribution function
#'   \item \code{qNLindley.HC()} — Quantile function
#'   \item \code{rNLindley.HC()} — Random generation
#'   \item \code{hNLindley.HC()} — Hazard function
#' }
#'
#' @usage
#' dNLindley.HC(x, lambda, theta, log = FALSE)
#' pNLindley.HC(q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qNLindley.HC(p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rNLindley.HC(n, lambda, theta)
#' hNLindley.HC(x, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dNLindley.HC}: numeric vector of (log-)densities
#'   \item \code{pNLindley.HC}: numeric vector of probabilities
#'   \item \code{qNLindley.HC}: numeric vector of quantiles
#'   \item \code{rNLindley.HC}: numeric vector of random variates
#'   \item \code{hNLindley.HC}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary, A.K. & Kumar, V. (2020). 
#' New Lindley Half Cauchy Distribution: Theory and Applications. 
#' \emph{International Journal of Recent Technology and Engineering (IJRTE)}, 
#' \bold{9(4)}, 1--7. \doi{10.35940/ijrte.D4734.119420}    
#'
#' @examples
#' x <- seq(1, 10, 0.5)
#' dNLindley.HC(x, 0.5, 1.5)
#' pNLindley.HC(x, 0.5, 1.5)
#' qNLindley.HC(0.5, 0.5, 1.5)
#' rNLindley.HC(10, 0.5, 1.5)
#' hNLindley.HC(x, 0.5, 1.5)
#' 
#' # Data
#' x <- reactorpump
#' # ML estimates
#' params = list(lambda=0.7743, theta=1.3829)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pNLindley.HC, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qNLindley.HC, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dNLindley.HC, pfun=pNLindley.HC, plot=TRUE)
#' print.gofic(out)
#' 
#' @name NewLindleyHC
#' @aliases dNLindley.HC pNLindley.HC qNLindley.HC rNLindley.HC hNLindley.HC
#' @export
NULL

#' @rdname NewLindleyHC
#' @usage NULL
#' @export   
dNLindley.HC <- function (x, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(lambda, theta) <= 0)) {
        stop("Parameters 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- (2/pi) * atan(x[valid]/lambda)
	w <- (theta * theta/(1 + theta))
	v <- lambda/(lambda * lambda + x[valid] * x[valid])
    pdf[valid] <- (2/pi) * w * v * (u ^ (theta-1)) * (1 - log(u))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname NewLindleyHC
#' @usage NULL
#' @export 
pNLindley.HC <- function (q, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(lambda, theta) <= 0)) {
		stop("Parameters 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- (2/pi)*atan(q[valid]/lambda)
    cdf[valid] <- (u ^ theta) * (1 - (theta/(1+theta)) * log(u))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname NewLindleyHC
#' @usage NULL
#' @export 
qNLindley.HC <- function (p, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Parameters 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	    theta*log((2/pi) * atan(x/lambda))+log(1-(theta/(theta+1))*log((2/pi) * atan(x/lambda))) -log(wi) 
	}
	
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
}

#' @rdname NewLindleyHC
#' @usage NULL
#' @export 
rNLindley.HC <- function(n, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(lambda, theta), is.numeric)))
		stop("'lambda', and 'theta' must be numeric.")
	if (any(c(lambda, theta) <= 0))
		stop("Invalid parameters: lambda, and theta must be positive.")

    u <- runif(n)
    q <- qNLindley.HC(u, lambda, theta)
    return(q)
}

#' @rdname NewLindleyHC
#' @usage NULL
#' @export 
hNLindley.HC <- function (x, lambda, theta)
{
    if (!all(sapply(list(x, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(lambda, theta) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dNLindley.HC(x, lambda, theta) 
    surv <- 1.0 - pNLindley.HC(x, lambda, theta)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 



