# PoissonGenRayleigh.R
#' Poisson Generalized Rayleigh (PGR) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the PGR distribution.
#'
#' The PGR distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The PGR distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1}{\left(1-e^{-\lambda}\right)}\left[1-\exp 
#' \left\{-\lambda\left(1-e^{-\beta x^2}\right)
#' ^\alpha\right\}\right]  \quad ;\;x > 0.
#' }
#'  
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dpois.gen.rayleigh()} — Density function
#'   \item \code{ppois.gen.rayleigh()} — Distribution function
#'   \item \code{qpois.gen.rayleigh()} — Quantile function
#'   \item \code{rpois.gen.rayleigh()} — Random generation
#'   \item \code{hpois.gen.rayleigh()} — Hazard function
#' }
#'
#' @usage
#' dpois.gen.rayleigh(x, alpha, beta, lambda, log = FALSE)
#' ppois.gen.rayleigh(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.gen.rayleigh(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.gen.rayleigh(n, alpha, beta, lambda)
#' hpois.gen.rayleigh(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.gen.rayleigh}: numeric vector of (log-)densities
#'   \item \code{ppois.gen.rayleigh}: numeric vector of probabilities
#'   \item \code{qpois.gen.rayleigh}: numeric vector of quantiles
#'   \item \code{rpois.gen.rayleigh}: numeric vector of random variates
#'   \item \code{hpois.gen.rayleigh}: numeric vector of hazard values
#' }
#'
#' @references
#' Joshi, R.K., & Kumar, V. (2021). 
#' Poisson Generalized Rayleigh Distribution with Properties and Application.
#' \emph{International Journal of Statistics and Applied Mathematics}, 
#' \bold{6(1)}, 90--99. \doi{10.22271/maths.2021.v6.i1b.637}    
#' 
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dpois.gen.rayleigh(x, 2.0, 0.5, 0.2)
#' ppois.gen.rayleigh(x, 2.0, 0.5, 0.2)
#' qpois.gen.rayleigh(0.5, 2.0, 0.5, 0.2)
#' rpois.gen.rayleigh(10, 2.0, 0.5, 0.2)
#' hpois.gen.rayleigh(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=1.5466, beta=0.0211, lambda=16.4523)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.gen.rayleigh, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.gen.rayleigh, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'             dfun = dpois.gen.rayleigh, pfun=ppois.gen.rayleigh, plot=TRUE)
#' print.gofic(out)
#' 
#' @name PoissonGenRayleigh
#' @aliases dpois.gen.rayleigh ppois.gen.rayleigh qpois.gen.rayleigh rpois.gen.rayleigh hpois.gen.rayleigh
#' @export
NULL

#' @rdname PoissonGenRayleigh
#' @usage NULL
#' @export  
dpois.gen.rayleigh <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))
	z <- 1- exp(-beta * x[valid] * x[valid])
    gx <- 2 * alpha * beta * x[valid] * exp(-beta * x[valid] * x[valid]) * (z ^ (alpha-1))	
    pdf[valid] <- tmp * gx * lambda * exp(-lambda * (z^alpha))
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonGenRayleigh
#' @usage NULL
#' @export   
ppois.gen.rayleigh <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))	
	gx <- (1- exp(-beta * q[valid] * q[valid]))^alpha
    cdf[valid] <- tmp * (1-exp(-lambda * gx))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonGenRayleigh
#' @usage NULL
#' @export   
qpois.gen.rayleigh <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	z <- -(1/lambda) * log(1-(1-exp(-lambda))*p[valid])		
    qtl[valid] <- (-(1/beta) * log(1-z^(1/alpha))) ^ 0.5   
    }    
  return(qtl)   
 } 

#' @rdname PoissonGenRayleigh
#' @usage NULL
#' @export  
rpois.gen.rayleigh <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.gen.rayleigh(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonGenRayleigh
#' @usage NULL
#' @export  
hpois.gen.rayleigh <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dpois.gen.rayleigh(x, alpha, beta, lambda)
    surv <- 1.0 - ppois.gen.rayleigh(x, alpha, beta, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 

