% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVG-package.R
\name{CalSVG}
\alias{CalSVG}
\title{Unified Interface for SVG Detection}
\usage{
CalSVG(
  expr_matrix,
  spatial_coords,
  method = c("meringue", "seurat", "binspect", "sparkx", "nnsvg", "markvario"),
  n_threads = 1L,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{expr_matrix}{Numeric matrix of gene expression values.
Rows are genes, columns are spatial locations (spots/cells).
Should be normalized (e.g., log-transformed counts).}

\item{spatial_coords}{Numeric matrix of spatial coordinates.
Rows are spatial locations, columns are x and y (and optionally z) coordinates.
Row names should match column names of \code{expr_matrix}.}

\item{method}{Character string specifying the SVG detection method.
One of: "meringue", "seurat", "binspect", "sparkx", "nnsvg", "markvario".}

\item{n_threads}{Integer. Number of threads for parallel computation.
Default is 1. Set to higher values for faster computation on multi-core systems.}

\item{verbose}{Logical. Whether to print progress messages. Default is TRUE.}

\item{...}{Additional arguments passed to the specific method function.}
}
\value{
A data.frame containing SVG detection results. The exact columns
  depend on the method used, but typically include:
  \itemize{
    \item \code{gene}: Gene identifiers
    \item \code{pval} or \code{p.value}: Raw p-values
    \item \code{padj} or \code{p.adj}: Adjusted p-values (multiple testing corrected)
    \item Method-specific statistics (e.g., Moran's I, LR statistic, odds ratio)
  }
}
\description{
A unified interface to run different spatially variable gene (SVG) detection
methods. This function provides a consistent API for all supported methods.
}
\details{
This function serves as a wrapper around the individual method functions:
\itemize{
  \item \code{method = "meringue"}: Calls \code{\link{CalSVG_MERINGUE}}
  \item \code{method = "seurat"}: Calls \code{\link{CalSVG_Seurat}}
  \item \code{method = "binspect"}: Calls \code{\link{CalSVG_binSpect}}
  \item \code{method = "sparkx"}: Calls \code{\link{CalSVG_SPARKX}}
  \item \code{method = "nnsvg"}: Calls \code{\link{CalSVG_nnSVG}}
  \item \code{method = "markvario"}: Calls \code{\link{CalSVG_MarkVario}}
}

For method-specific parameters, please refer to the documentation of individual
method functions.
}
\examples{
\donttest{
# Simulate example data
set.seed(42)
n_genes <- 20
n_spots <- 100
expr_matrix <- matrix(rpois(n_genes * n_spots, lambda = 10),
                      nrow = n_genes, ncol = n_spots)
rownames(expr_matrix) <- paste0("gene_", seq_len(n_genes))
colnames(expr_matrix) <- paste0("spot_", seq_len(n_spots))

spatial_coords <- cbind(x = runif(n_spots, 0, 100),
                        y = runif(n_spots, 0, 100))
rownames(spatial_coords) <- colnames(expr_matrix)

# Run SPARK-X method (no external dependencies)
results <- CalSVG(expr_matrix, spatial_coords, method = "sparkx",
                  kernel_option = "single", verbose = FALSE)
head(results)
}

}
\seealso{
\code{\link{CalSVG_MERINGUE}}, \code{\link{CalSVG_binSpect}},
\code{\link{CalSVG_SPARKX}}, \code{\link{CalSVG_nnSVG}}
}
