% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unit_root.R
\name{test_unit_root}
\alias{test_unit_root}
\title{Comprehensive Unit Root Test Suite}
\usage{
test_unit_root(y, max_lags = NULL, significance_level = 0.05, verbose = FALSE)
}
\arguments{
\item{y}{Numeric vector of the time series to test.}

\item{max_lags}{Maximum number of lags for ADF-type tests. If NULL,
defaults to \code{floor(12 * (length(y)/100)^0.25)}.}

\item{significance_level}{Significance level for hypothesis testing.
Default is 0.05.}

\item{verbose}{Logical indicating whether to print detailed results.}
}
\value{
A list of class "signaly_unitroot" containing:
\describe{
\item{adf}{Results from ADF tests (none, drift, trend specifications)}
\item{ers}{Results from ERS tests (DF-GLS and P-test)}
\item{kpss}{Results from KPSS tests (level and trend)}
\item{pp}{Results from Phillips-Perron tests}
\item{summary}{Data frame summarizing all test results}
\item{conclusion}{Integrated conclusion about stationarity}
\item{persistence_type}{Classification: stationary, trend-stationary,
difference-stationary, or inconclusive}
}
}
\description{
Applies multiple unit root and stationarity tests to a time series,
providing an integrated assessment of persistence properties. Implements
Augmented Dickey-Fuller (ADF), Elliott-Rothenberg-Stock (ERS),
Kwiatkowski-Phillips-Schmidt-Shin (KPSS), and Phillips-Perron tests.
}
\details{
The battery of tests addresses different null hypotheses and specifications:

\strong{Augmented Dickey-Fuller (ADF)} tests the null of a unit root against
the alternative of stationarity. Three specifications are tested:
\itemize{
\item \strong{none}: No constant, no trend (random walk)
\item \strong{drift}: Constant included (random walk with drift)
\item \strong{trend}: Constant and linear trend
}

\strong{Elliott-Rothenberg-Stock (ERS)} tests provide more power than ADF by
using GLS detrending. Two variants:
\itemize{
\item \strong{DF-GLS}: GLS-detrended Dickey-Fuller test
\item \strong{P-test}: Point-optimal test
}

\strong{KPSS} reverses the hypotheses: null is stationarity, alternative is
unit root. This allows testing the stationarity hypothesis directly.

\strong{Phillips-Perron} uses non-parametric corrections for serial correlation,
avoiding lag selection issues.
}
\section{Interpretation Strategy}{

The function synthesizes results using the following logic:

\enumerate{
\item If ADF/ERS reject unit root AND KPSS fails to reject stationarity:
Series is likely \strong{stationary}
\item If ADF/ERS fail to reject AND KPSS rejects stationarity:
Series likely has \strong{unit root} (difference-stationary)
\item If only trend-ADF rejects: Series is likely \strong{trend-stationary}
\item Conflicting results indicate \strong{inconclusive} or structural breaks
}
}

\examples{
set.seed(123)
stationary <- arima.sim(list(ar = 0.5), n = 100)
result <- test_unit_root(stationary)
print(result$conclusion)

nonstationary <- cumsum(rnorm(100))
result2 <- test_unit_root(nonstationary)
print(result2$conclusion)

}
\references{
Dickey, D. A., & Fuller, W. A. (1979). Distribution of the Estimators for
Autoregressive Time Series with a Unit Root. Journal of the American
Statistical Association, 74(366), 427-431.

Elliott, G., Rothenberg, T. J., & Stock, J. H. (1996). Efficient Tests for
an Autoregressive Unit Root. Econometrica, 64(4), 813-836.

Kwiatkowski, D., Phillips, P. C. B., Schmidt, P., & Shin, Y. (1992).
Testing the null hypothesis of stationarity against the alternative of a
unit root. Journal of Econometrics, 54(1-3), 159-178.

Phillips, P. C. B., & Perron, P. (1988). Testing for a unit root in time
series regression. Biometrika, 75(2), 335-346.
}
\seealso{
\code{\link[urca]{ur.df}}, \code{\link[urca]{ur.ers}},
\code{\link[urca]{ur.kpss}}, \code{\link[urca]{ur.pp}}
}
