/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

H5FL_DEFINE_STATIC(H5EA_sblock_t);

H5FL_SEQ_DEFINE_STATIC(haddr_t);

H5FL_BLK_DEFINE(page_init);

H5EA_sblock_t *
H5EA__sblock_alloc(H5EA_hdr_t *hdr, H5EA_iblock_t *parent, unsigned sblk_idx)
{
    H5EA_sblock_t *sblock    = NULL; 
    H5EA_sblock_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (sblock = H5FL_CALLOC(H5EA_sblock_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array super block");

    
    if (H5EA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    sblock->hdr = hdr;

    
    sblock->parent = parent;
    sblock->addr   = HADDR_UNDEF;

    
    sblock->idx    = sblk_idx;
    sblock->ndblks = hdr->sblk_info[sblk_idx].ndblks;
    assert(sblock->ndblks);
    sblock->dblk_nelmts = hdr->sblk_info[sblk_idx].dblk_nelmts;

    
    if (NULL == (sblock->dblk_addrs = H5FL_SEQ_MALLOC(haddr_t, sblock->ndblks)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for super block data block addresses");

    
    if (sblock->dblk_nelmts > hdr->dblk_page_nelmts) {
        
        sblock->dblk_npages = sblock->dblk_nelmts / hdr->dblk_page_nelmts;

        
        assert(sblock->dblk_npages > 1);

        
        assert((sblock->dblk_npages * hdr->dblk_page_nelmts) == sblock->dblk_nelmts);

        
        sblock->dblk_page_init_size = ((sblock->dblk_npages) + 7) / 8;
        assert(sblock->dblk_page_init_size > 0);

        
        if (NULL ==
            (sblock->page_init = H5FL_BLK_CALLOC(page_init, sblock->ndblks * sblock->dblk_page_init_size)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for super block page init bitmask");

        
        sblock->dblk_page_size = (hdr->dblk_page_nelmts * hdr->cparam.raw_elmt_size) + H5EA_SIZEOF_CHKSUM;
    } 

    
    ret_value = sblock;

done:
    if (!ret_value)
        if (sblock && H5EA__sblock_dest(sblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array super block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5EA__sblock_create(H5EA_hdr_t *hdr, H5EA_iblock_t *parent, bool *stats_changed, unsigned sblk_idx)
{
    H5EA_sblock_t *sblock = NULL;           
    haddr_t        sblock_addr;             
    haddr_t        tmp_addr  = HADDR_UNDEF; 
    bool           inserted  = false;       
    haddr_t        ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(stats_changed);

    
    if (NULL == (sblock = H5EA__sblock_alloc(hdr, parent, sblk_idx)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for extensible array super block");

    
    sblock->size = H5EA_SBLOCK_SIZE(sblock);

    
    sblock->block_off = hdr->sblk_info[sblk_idx].start_idx;

    
    if (HADDR_UNDEF == (sblock_addr = H5MF_alloc(hdr->f, H5FD_MEM_EARRAY_SBLOCK, (hsize_t)sblock->size)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "file allocation failed for extensible array super block");
    sblock->addr = sblock_addr;

    
    H5VM_array_fill(sblock->dblk_addrs, &tmp_addr, sizeof(haddr_t), sblock->ndblks);

    
    if (H5AC_insert_entry(hdr->f, H5AC_EARRAY_SBLOCK, sblock_addr, sblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINSERT, HADDR_UNDEF,
                    "can't add extensible array super block to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, sblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add extensible array entry as child of array proxy");
        sblock->top_proxy = hdr->top_proxy;
    } 

    
    hdr->stats.stored.nsuper_blks++;
    hdr->stats.stored.super_blk_size += sblock->size;

    
    *stats_changed = true;

    
    ret_value = sblock_addr;

done:
    if (!H5_addr_defined(ret_value))
        if (sblock) {
            
            if (inserted)
                if (H5AC_remove_entry(sblock) < 0)
                    HDONE_ERROR(H5E_EARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove extensible array super block from cache");

            
            if (H5_addr_defined(sblock->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_EARRAY_SBLOCK, sblock->addr, (hsize_t)sblock->size) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to release extensible array super block");

            
            if (H5EA__sblock_dest(sblock) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to destroy extensible array super block");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_sblock_t *
H5EA__sblock_protect(H5EA_hdr_t *hdr, H5EA_iblock_t *parent, haddr_t sblk_addr, unsigned sblk_idx,
                     unsigned flags)
{
    H5EA_sblock_t         *sblock = NULL; 
    H5EA_sblock_cache_ud_t udata;         
    H5EA_sblock_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(sblk_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.hdr       = hdr;
    udata.parent    = parent;
    udata.sblk_idx  = sblk_idx;
    udata.sblk_addr = sblk_addr;

    
    if (NULL ==
        (sblock = (H5EA_sblock_t *)H5AC_protect(hdr->f, H5AC_EARRAY_SBLOCK, sblk_addr, &udata, flags)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect extensible array super block, address = %llu",
                    (unsigned long long)sblk_addr);

    
    if (hdr->top_proxy && NULL == sblock->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, sblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, NULL,
                        "unable to add extensible array entry as child of array proxy");
        sblock->top_proxy = hdr->top_proxy;
    }

    
    ret_value = sblock;

done:
    
    if (!ret_value) {
        
        if (sblock &&
            H5AC_unprotect(hdr->f, H5AC_EARRAY_SBLOCK, sblock->addr, sblock, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect extensible array super block, address = %llu",
                        (unsigned long long)sblock->addr);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__sblock_unprotect(H5EA_sblock_t *sblock, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(sblock);

    
    if (H5AC_unprotect(sblock->hdr->f, H5AC_EARRAY_SBLOCK, sblock->addr, sblock, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect extensible array super block, address = %llu",
                    (unsigned long long)sblock->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__sblock_delete(H5EA_hdr_t *hdr, H5EA_iblock_t *parent, haddr_t sblk_addr, unsigned sblk_idx)
{
    H5EA_sblock_t *sblock = NULL; 
    size_t         u;             
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(sblk_addr));

    
    if (NULL == (sblock = H5EA__sblock_protect(hdr, parent, sblk_addr, sblk_idx, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array super block, address = %llu",
                    (unsigned long long)sblk_addr);

    
    for (u = 0; u < sblock->ndblks; u++) {
        
        if (H5_addr_defined(sblock->dblk_addrs[u])) {
            
            if (H5EA__dblock_delete(hdr, sblock, sblock->dblk_addrs[u], sblock->dblk_nelmts) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL, "unable to delete extensible array data block");
            sblock->dblk_addrs[u] = HADDR_UNDEF;
        } 
    }     

done:
    
    if (sblock && H5EA__sblock_unprotect(sblock, H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG |
                                                     H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array super block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__sblock_dest(H5EA_sblock_t *sblock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(sblock);
    assert(!sblock->has_hdr_depend);

    
    if (sblock->hdr) {
        
        if (sblock->dblk_addrs)
            sblock->dblk_addrs = H5FL_SEQ_FREE(haddr_t, sblock->dblk_addrs);

        
        if (sblock->page_init) {
            assert(sblock->dblk_npages > 0);
            sblock->page_init = H5FL_BLK_FREE(page_init, sblock->page_init);
        } 

        
        if (H5EA__hdr_decr(sblock->hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        sblock->hdr = NULL;
    } 

    
    assert(NULL == sblock->top_proxy);

    
    sblock = H5FL_FREE(H5EA_sblock_t, sblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
