% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roots.R
\name{roots_bisec}
\alias{roots_bisec}
\title{Bisection method for roots}
\usage{
roots_bisec(
  fn,
  x0 = 0,
  lB = NULL,
  rB = NULL,
  tol = 1e-09,
  imax = 1e+06,
  eps = 0.1,
  message = TRUE,
  logg = FALSE,
  ...
)
}
\arguments{
\item{fn}{A function of one variable. If the function includes variables,
these will have to be passed as additional variables, using the
same names in the function's definition (see examples).}

\item{x0}{A numeric variable. The starting value to find the initial
interval, when a search interval is not provided. The default
value is `x0=0`.}

\item{lB}{A numeric variable indicating the lower (left) extreme of 
the search interval. If not given, this number will be selected
starting from `x0` and in small steps `eps` of values
smaller than `x0`, until a value of `lB` is found for
which the function `f` has sign opposite of the sign it
has at `rB`. Default is for `lB` not to be entered (`lB=NULL`).}

\item{rB}{Same as `lB`, but corresponding to the upper (right) extreme of
the search interval. Default is for `rB` not to be entered 
(`rB=NULL`).}

\item{tol}{A real number, in general a small number. The width of the
smallest interval containing the zero of the function just
before the algorithm stops. This means that the largest error
\eqn{|x-x_t|} between the numerical value of the root found, 
\eqn{x}, and its correct value, \eqn{x_t}, is tol. Default
value is 1e-9.}

\item{imax}{A positive integer. The maximum number of bisections of the
interval, while searching the zero of the function. The
default value is 1e6, although convergence is normally 
obtained with a number of bisections much smaller than `imax`.
`imax` is important to stop search in those cases in which the
function has no zeros in the search interval provided.}

\item{eps}{A real number. The step size needed for the selection of a
search interval, when this is not provided. In such a 
situation, symmetric intervals with increasing width around
`x0` are considered where the left and right extremes are 
`x0-i*eps` and `x0+i*eps`, respectively, where `i` is a positive
integer, progressively increasing from 1 to the maximum
allowed value `imax`. Search for the selected interval stops
when is the signs of the function `f` calculated at the extremes 
are opposite. If the search interval is not found, a warning
message is printed and NULL is returned. Default value is
0.1.}

\item{message}{A logical variable to state whether messages about the 
root and the largest error have to be printed. The default
is for the messages to be printed (message=TRUE).}

\item{logg}{A logical variable to state whether information on the series
of bisected intervals is printed (TRUE) or not (FALSE). Default
is for such information not to be printed (FALSE).}

\item{...}{Parameters passed to function `fn`, if needed.}
}
\value{
A numeric value, the zero of the function (or, equivalently, the
        root of the equation \eqn{f(x)=0}).
}
\description{
Find the zero of a function of one variable, \eqn{f(x)}, given a starting
value close to the zero, or an interval including the zero.
}
\details{
Finding the zero of \eqn{f(x)} is equivalent to finding the roots of the
equation:
\deqn{
 f(x) = 0 
}
The algorithm used is based on the bisection method that needs an initial
interval within which the root is supposed to reside. When multiple roots
are involved, the algorithm will only find one among those inside the
chosen interval. The algorithm can be started also with just one value,
x0, supposedly close to the wanted root. In this case, an interval
is selected so that the function at the extremes of the interval has
opposite signs. If such an interval is not found, the function dumps a
warning message and returns NULL. The bisection method has a slow 
convergence rate and it does not converge at all in specific situations.
}
\examples{
# The quadratic equation x^2-5*x+6=0 has two roots, 2 and 3
ff <- function(x) return(x^2-5*x+6)

# Find root 2, starting from a single point close to 2
x0 <- 1
x <- roots_bisec(ff,x0=1)
print(x)

# Find root 3, using an interval (no message printing)
x <- roots_bisec(ff,lB=2.8,rB=4,message=FALSE)
print(x)

# Function with a parameter f(x) = exp(x) - k
ff <- function(x,k=2) return(exp(x)-k)

# Solution of exp(x)=3 is log(3)
x <- roots_bisec(ff,k=3)
print(log(3))

}
