#' Create a Combined gtsummary Table and Forest Plot
#'
#' This is the main wrapper function that takes a 'gtsummary' object,
#' converts it to a 'ggplot' table, extracts the necessary data, creates
#' a forest plot, and combines the two plots side-by-side using `+`.
#' This likely relies on the `patchwork` package for plot combination.
#'
#' @param tbl (`gtsummary`)\cr
#'   A 'gtsummary' object (e.g., from [tbl_regression()]).
#'
#' @return A combined 'ggplot' object (likely a 'patchwork' object)
#'   showing the table on the left and the forest plot on the right.
#'
#' @seealso [extract_plot_data()], [gg_forest_plot()]
#'
#' @export
#' @examplesIf identical(Sys.getenv("NOT_CRAN"), "true")
#' tbl <-
#'   trial |>
#'   tbl_roche_subgroups(
#'     rsp = "response",
#'     by = "trt",
#'     subgroups = c("grade", "stage"),
#'     ~ glm(response ~ trt, data = .x) |>
#'       gtsummary::tbl_regression(
#'         show_single_row = trt,
#'         exponentiate = TRUE
#'       )
#'   )
#' \dontrun{
#' g_forest(tbl)
#' }
g_forest <- function(tbl) {
  # todo need to make sure tbl does not have wrapped rows
  suppressMessages(
    table_plot <- as_ggplot(tbl)
  )
  forest_data <- extract_plot_data(tbl)
  forest_header <- attr(tbl, "by")
  forest_plot <- gg_forest_plot(forest_data, forest_header)
  table_plot + forest_plot + plot_layout(widths = c(3, 1))
}


#' Create a Custom Forest Plot
#'
#' Generates a forest plot using `ggplot2` from a data frame containing
#' estimates, confidence intervals, and sample sizes. This function is designed
#' to be a component of a combined table/plot output (e.g., used by [g_forest()]).
#'
#' @param data (`data.frame`)\cr
#'   A data frame (tibble) containing the plot data. It must include
#'   columns: `estimate`, `ci_lower`,
#'   `ci_upper`, and `n` (for point size).
#' @param header Forest header
#' @param xlim (`numeric(2)`)\cr
#'   A numeric vector of length 2 specifying the limits of the x-axis
#'   (e.g., `c(0.1, 10)`).
#' @param logx (`logical(1)`)\cr
#'   A logical value indicating whether the x-axis should be log-transformed
#'   (i.e., using [scale_x_log10()]). The default is `TRUE`, which is typical
#'   for effect measures like Odds Ratios or Hazard Ratios.
#' @param vline (`numeric(1)`)\cr
#'   A numeric value specifying the x-intercept for the vertical
#'   reference line (line of no effect). The default is `1`.
#'
#' @return A 'ggplot' object representing the forest plot.
#' @keywords internal
#'
#' @examples
#' \dontrun{
#' # Assuming 'forest_data' is structured correctly:
#' forest_data <- data.frame(
#'   estimate = c(0.5, 2.0),
#'   ci_lower = c(0.2, 1.5),
#'   ci_upper = c(0.9, 3.5),
#'   n = c(100, 250)
#' )
#'
#' gg_forest_plot(forest_data)
#' gg_forest_plot(forest_data, xlim = c(0.05, 50), vline = 1)
#' }
gg_forest_plot <- function(data,
                           header = "",
                           xlim = c(0.1, 10),
                           logx = TRUE,
                           vline = 1) {
  forest_header <- paste0(header, "\nBetter")
  # Calculate y positions (reverse order for top-to-bottom display)
  data <- data |>
    mutate(y_pos = rev(dplyr::row_number()))

  # Apply log transformation if needed
  if (logx) {
    data <- data |>
      mutate(
        estimate_log = .data$estimate,
        ci_lower_log = .data$ci_lower,
        ci_upper_log = .data$ci_upper
      )
    x_aesthetic_vars <- aes(x = .data$estimate, xend = .data$ci_lower_log, yend = .data$ci_upper_log)
    x_scale <- scale_x_log10(expand = c(0.01, 0))
  }

  # Create plot
  ggplot(data) +
    # Background rectangle
    annotate("rect",
      xmin = xlim[1], xmax = xlim[2], ymin = 0.5, ymax = nrow(data) + 0.5,
      fill = "grey92", alpha = 0.5
    ) +
    # CI lines with arrows
    geom_errorbar(aes(xmin = .data$ci_lower_log, xmax = .data$ci_upper_log, y = .data$y_pos),
      width = 0.2, color = "black", orientation = "y"
    ) +
    # Points
    geom_point(aes(x = .data$estimate_log, y = .data$y_pos, size = .data$n),
      color = "#343cff", shape = 19
    ) +
    # Reference line
    geom_vline(xintercept = vline, linewidth = 1) +
    # Forest header text
    annotate("text",
      # Use geometric mean for log-scale centering: exp(mean(log(c(x1, x2))))
      x = 10^(mean(log10(c(xlim[1], vline)))),
      y = nrow(data) + 1.25,
      label = forest_header[1],
      size = 3.5,
      hjust = 0.5 # Now it will look truly centered
    ) +
    annotate("text",
      x = 10^(mean(log10(c(vline, xlim[2])))),
      y = nrow(data) + 1.25,
      label = forest_header[2],
      size = 3.5,
      hjust = 0.5
    ) +
    # Scales and theme
    x_scale +
    coord_cartesian(xlim = xlim) +
    scale_y_continuous(
      limits = c(0, nrow(data) + 2.5), breaks = data$y_pos,
      labels = rep("", length(data$estimate)), expand = c(0, 0)
    ) +
    theme_minimal() +
    theme(
      panel.grid.major = element_blank(),
      panel.grid.minor = element_blank(),
      axis.title.y = element_blank(),
      axis.title.x = element_blank(),
      legend.position = "none",
      plot.margin = margin(0.1, 0.1, 0.05, 0, "npc")
    )
}

#' Extract Data for Forest Plot from gtsummary Table
#'
#' Converts the table body (`tbl$table_body`) of a 'gtsummary' object
#' into a data frame suitable for plotting with [gg_forest_plot()].
#' It selects and renames the necessary columns for the plot.
#'
#' @param tbl (`gtsummary`)\cr
#'   A 'gtsummary' object (e.g., from [tbl_regression()] or [tbl_uvregression()]).
#'
#' @return A data frame (tibble) with the columns:
#'   `group` (from `term`), `estimate`, `ci_lower` (from `conf.low`),
#'   `ci_upper` (from `conf.high`), and `n` (from `N_obs`).
#' @keywords internal
extract_plot_data <- function(tbl) {
  ret <- tbl$table_body |>
    select(
      # group = termc("Drug A", "Drug B"),
      estimate = starts_with("estimate"),
      ci_lower = starts_with("conf.low"),
      ci_upper = starts_with("conf.high"),
      n = starts_with("N_obs", ignore.case = FALSE)
    )
  return(ret)
}
