#' Calculate state expectancy
#'
#' @description
#' This function calculates the expected time spent in the transient states
#' (state expectancy).
#'
#' @details
#' If the argument `start_distr` is specified, the average of the state
#' expectancies over all starting states is calculated. The names and length
#' of `start_distr` need to match the starting states generated by this
#' function which are based on the `dtms` object.
#'
#' The partial expectancy for the time spent in the transient states can be
#' calculated using the arguments `start_time` and `end_time`.
#'
#' IF the argument `risk` is specified, then only the remaining life expectancy
#' for the state specified with this argument is shown, but for all time units
#' of the time scale.
#'
#' Two corrections to the results will be applied per default. Both corrections
#' are required as the underlying formulas do actually not provide the
#' expected time spent in a state, but the number of visits to a state. Time
#' and visits are only equal under certain conditions; in particular, only if
#' transitions between states happen mid-interval and the step length of the
#' time scale is equal to one. The first correction will remove a certain amount
#' of time spent in a certain state if its equal to the starting state. This is
#' controlled with the argument `correction` which is applied multiplicative.
#' For instance, its default value 0.5 means that the state expectancy in some
#' state X starting from state X is reduced by 0.5 time steps. The second
#' correction uses the entry `timestep` of `dtms`, and multiplies results with its value.
#'
#' @param probs Data frame with transition probabilities, as created with \code{dtms_transitions}.
#' @param dtms dtms object, as created with \code{dtms}.
#' @param matrix Matrix with transition probabilities, as generated with \code{dtms_matrix}.
#' @param risk Character (otpional), name of one transient state. If specified expectancies are only shown for this state but by values of the time scale.
#' @param start_distr Numeric (optional), distribution of starting states. If specified, average expectancy over all starting states will be calculated. Only applied if risk=NULL.
#' @param start_state Character (optional), name of starting states. If NULL (default) all transient states will be used.
#' @param start_time Numeric (optional), value of time scale for start. If NULL (default) first value of time scale will be used.
#' @param end_time Numeric (optional), last value of time scale to consider. If NULL (default) all values of time scale starting from start_time will be used.
#' @param correction Numeric (optional), correction for expectancy when starting state and state under consideration match, see details. Defaults to 0.5.
#' @param total Logical (optional), calculate total expectancy. Default is TRUE. Only applied if risk=NULL.
#' @param verbose Logical (optional), print some information on what is computed. Default is FALSE.
#' @param fundamental Logical (optional), return fundamental matrix? Default is FALSE.
#'
#' @return A matrix with state expectancies.
#' @export
#'
#' @examples
#' ## Define model: Absorbing and transient states, time scale
#' simple <- dtms(transient=c("A","B"),
#'                absorbing="X",
#'                timescale=0:20)
#' ## Reshape to transition format
#' estdata <- dtms_format(data=simpledata,
#'                        dtms=simple,
#'                        idvar="id",
#'                        timevar="time",
#'                        statevar="state")
#' ## Clean
#' estdata <- dtms_clean(data=estdata,
#'                       dtms=simple)
#' # Fit model
#' fit <- dtms_fit(data=estdata)
#' ## Predict probabilities
#' probs    <- dtms_transitions(dtms=simple,
#'                              model = fit)
#' ## Get starting distribution
#' S <- dtms_start(dtms=simple,
#'                 data=estdata)
#' ## State expectancies
#' dtms_expectancy(dtms=simple,
#'                 probs=probs,
#'                 start_distr=S)

dtms_expectancy <- function(probs=NULL,
                            matrix=NULL,
                            dtms,
                            risk=NULL,
                            start_distr=NULL,
                            start_time=NULL,
                            start_state=NULL,
                            end_time=NULL,
                            correction=0.5,
                            total=TRUE,
                            fundamental=FALSE,
                            verbose=FALSE) {

  # Check
  dtms_proper(dtms)

  # Get matrix if not specified
  if(is.null(matrix)) matrix <- dtms_matrix(probs=probs,
                                            dtms=dtms)

  # Starting state and time
  if(is.null(start_state)) start_state <- dtms$transient
  if(is.null(start_time)) start_time <- min(dtms$timescale)

  # Starting states, long names
  starting <- dtms_combine(start_state,start_time,sep=dtms$sep)

  # Number of starting and receiving states
  nstart <- length(starting)
  ntransient <- length(dtms$transient)

  # Remove absorbing states
  matrix <- dtms_absorbing(matrix)

  # All transient states
  allstates <- rownames(matrix)

  # Fundamental matrix
  nstates <- dim(matrix)[1]
  Nmat <- solve(diag(1,nstates)-matrix)

  # Correction
  if(is.numeric(correction)) {

    # Adjust
    diag(Nmat) <- diag(Nmat) - correction

    # Output
    if(verbose) cat("(Applying correction)","\n\n")
  }

  # Only return fundamental matrix?
  if(fundamental) {
    return(Nmat)
  }

  # Variant 1: Expectation of all transient states
  if(is.null(risk)) {

    # Matrix for results
    result <- matrix(data=NA,ncol=ntransient,nrow=nstart)
    rownames(result) <- paste0("start:",starting)
    colnames(result) <- dtms$transient

    for(i in 1:ntransient) {

      # Get states
      selector <- dtms_in(allstates,dtms$transient[i],dtms$sep)

      # Use end_time if specified
      if(!is.null(end_time)) {
        times <- dtms_gettime(allstates,dtms$sep)
        times <- times<=end_time
        times[!is.logical(times)] <- FALSE
        selector <- selector & times
      }

      # Calculate results and place
      if(nstart>1) tmp <- rowSums(Nmat[starting,selector]) else tmp <- sum(Nmat[starting,selector])

      # Place
      result[,dtms$transient[i]] <- tmp
    }

  }

  # Variant 2: Expectation in one state by time scale
  if(!is.null(risk)) {

    # Check
    if(length(risk)!=1) stop("Only one state allowed for 'risk'")

    # Get time right
    first <- which(dtms$timescale==start_time)
    if(is.null(end_time)) last <- length(dtms$timescale) else
      last <- which(dtms$timescale==end_time)
    times <- dtms$timescale[first:last]
    ntimes <- length(times)

    # Get right columns from fundamental matrix
    selector1 <- dtms_in(allstates,risk,dtms$ep)
    selector2 <- dtms_gettime(allstates,dtms$sep)%in%times
    selector <- selector1 & selector2

    # Get result
    tmp <- rowSums(Nmat[,selector])

    # Matrix with results
    result <- matrix(data=tmp,ncol=ntimes,nrow=nstart,byrow=TRUE)
    rownames(result) <- paste0("start:",starting)
    colnames(result) <- paste(times)

  }

  # Calculate average if starting distribution is provided
  if(!is.null(start_distr) & is.null(risk)) {

    # Check if matching
    if(length(start_distr)!=dim(result)[1]) stop("Starting distribution too long or short")

    # Match to starting/row ordering of result
    start_distr <- start_distr[match(names(start_distr),starting)]

    # Calculate
    AVERAGE <- colSums(result*start_distr)

    # Put into matrix for results
    result <- rbind(result,AVERAGE)
  }

  # Add row totals
  if(total & is.null(risk)) {
    TOTAL <- rowSums(result)
    result <- cbind(result,TOTAL)
  }

  # Adjust for time step
  if(dtms$timestep!=1) {
    result <- result*dtms$timestep
    if(verbose) cat("Adjusting for step length","\n\n")
  }

  # Return result
  return(result)

}
