% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distributions.R
\name{dfr_weibull}
\alias{dfr_weibull}
\title{Weibull Distribution (Power-Law Hazard)}
\usage{
dfr_weibull(shape = NULL, scale = NULL)
}
\arguments{
\item{shape}{Shape parameter (k). Controls hazard behavior:
k < 1: decreasing hazard (infant mortality)
k = 1: constant hazard (exponential)
k > 1: increasing hazard (wear-out)}

\item{scale}{Scale parameter (sigma). Controls time scale.}
}
\value{
A \code{dfr_dist} object with analytical rate, cumulative hazard,
and score function.
}
\description{
Creates a DFR distribution with Weibull hazard function.
The Weibull is extremely versatile: it can model increasing (wear-out),
decreasing (infant mortality), or constant (exponential) failure rates.
}
\details{
The Weibull distribution has:
\itemize{
\item Hazard: \eqn{h(t) = (k/\sigma)(t/\sigma)^{k-1}}
\item Cumulative hazard: \eqn{H(t) = (t/\sigma)^k}
\item Survival: \eqn{S(t) = e^{-(t/\sigma)^k}}
\item Characteristic life (63.2\% failure): \eqn{\sigma}
}
}
\section{Reliability Interpretation}{

\itemize{
\item Shape < 1: Infant mortality (burn-in failures, defects)
\item Shape = 1: Random failures (reduces to exponential)
\item Shape = 2: Rayleigh distribution (linear hazard increase)
\item Shape > 2: Accelerating wear-out (fatigue, corrosion)
}
}

\section{B-Life Calculation}{

The B10 life (10\% failure quantile) is commonly used in reliability:
\eqn{B10 = \sigma \cdot (-\log(0.9))^{1/k}}
}

\examples{
# Bearing with wear-out failure (shape > 1)
bearing <- dfr_weibull(shape = 2.5, scale = 50000)

# Hazard increases with time
h <- hazard(bearing)
h(10000)  # hazard at 10k hours
h(40000)  # much higher at 40k hours

# B10 life calculation
Q <- inv_cdf(bearing)
B10 <- Q(0.10)  # 10\% failure quantile

# Fit to test data with right-censoring
set.seed(123)
test_data <- data.frame(
  t = pmin(rweibull(100, shape = 2.5, scale = 50000), 30000),
  delta = as.integer(rweibull(100, shape = 2.5, scale = 50000) <= 30000)
)
solver <- fit(dfr_weibull())
result <- solver(test_data, par = c(2, 40000))
coef(result)

}
